% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/discrete_inhibit_sample.r
\name{discrete.inhibit.sample}
\alias{discrete.inhibit.sample}
\title{Spatially discrete sampling}
\usage{
discrete.inhibit.sample(obj, size, delta, delta.fix = FALSE, k = 0,
  cp.criterion = NULL, zeta, ntries = 10000, poly = NULL,
  plotit = TRUE)
}
\arguments{
\item{obj}{a \code{sf} or \code{sp} object where each line corresponds to a spatial location containing values of two-dimensional coordinates and, optionally, the values of one or more associated values, typically an outcome of interest and any associated covariates.}

\item{size}{a non-negative integer giving the total number of locations to be sampled.}

\item{delta}{minimum permissible distance between any two locations in preliminary sample. This can be allowed to vary with number of \code{'close pairs'} if a \bold{simple inhibitory} design is compared to one of the \bold{inhibitory plus close pairs} design.}

\item{delta.fix}{'logical' specifies whether \code{'delta'} is fixed or allowed to vary with number of close pairs \eqn{k}. Default is \code{delta.fix = FALSE}.}

\item{k}{number of close-pair locations in the sample. Must be an integer between 0 and \code{size}/2.}

\item{cp.criterion}{criterion for choosing close pairs \eqn{k}. The \code{"cp.zeta"} criterion chooses locations not included in the initial sample, from the uniform distribution of a disk with radius \code{'zeta'} (NB: \code{zeta} argument must be provided for this criterion). The \code{"cp.neighb"} criterion chooses nearest neighbours amongst locations not included in the initial sample (\code{'zeta'} becomes trivial for \code{'cp.neighb'} criterion).}

\item{zeta}{maximum permissible distance (radius of a disk with center \eqn{x^{*}_{j}, j = 1, \ldots, k}) within which a close-pair point is placed. See \bold{Details}.}

\item{ntries}{number of rejected proposals after which the algorithm terminates.}

\item{poly}{'optional', a \code{sf} or \code{sp} polygon object in which the design sits. The default is the bounding box of points given by \code{obj}.}

\item{plotit}{'logical' specifying if graphical output is required. Default is \code{plotit = TRUE}.}
}
\value{
a list with the following four components:

\code{unique.locs:} the number of unique sampled locations.

\code{delta:} the value of \eqn{\delta} after taking into account the number of close pairs \eqn{k}. If \code{delta.fix = TRUE}, this will be \eqn{\delta} input by the user.

\eqn{k:} the number of close pairs included in the sample (for \bold{inhibitory plus close pairs} design).

\code{sample.locs:} a \code{sf} or \code{sp} object containing the final sampled locations and any associated values.
}
\description{
Draw a spatially discrete sample from a specified set of spatial locations within a polygonal sampling region according to an \bold{"inhibitory plus close pairs"} specification.
}
\details{
To draw a sample of size \eqn{n} from a population of spatial locations \eqn{X_{i} : i  = 1,\ldots,N}, with the property that the distance between any two sampled locations is at least \eqn{\delta}, the function implements the following algorithm.
\itemize{
\item{Step 1.} Draw an initial sample of size \eqn{n}  completely at random and call this \eqn{x_{i}  : i  = 1,\dots, n}.
\item{Step 2.} Set \eqn{i  = 1}.
\item{Step 3.} Calculate the smallest distance, \eqn{d_{\min}}, from \eqn{x_{i}}  to all other \eqn{x_{j}}  in the initial sample.
\item{Step 4.} If \eqn{d_{\min} \ge \delta}, increase \eqn{i}  by 1 and return to step 2 if \eqn{i \le n}, otherwise stop.
\item{Step 5.} If \eqn{d_{\min} < \delta}, draw an integer \eqn{j}  at random from \eqn{1,  2,\ldots,N}, set \eqn{x_{i}  = X_{j}}  and return to step 3.}

Samples generated in this way exhibit  more regular spatial arrangements than would random samples of the same size. The degree of regularity achievable will be influenced by the spatial arrangement of the population \eqn{X_{i}  : i  = 1,\ldots,N}, the specified value of \eqn{\delta}  and the sample size \eqn{n}. For any given population, if \eqn{n}  and/or \eqn{\delta} is too large, a sample of the required size with the distance between any two sampled locations at least \eqn{\delta} will not be achievable; the algorithm will then find \eqn{n_{s} < n} points that can be placed for the given parameters.

\bold{Sampling close pairs of points.}

For some purposes, typically when using the same sample for parameter estimation and spatial prediction, it is desirable that a spatial sampling scheme include pairs of closely spaced points \eqn{x}. The function offers two ways of specifying close pairs, either as the closest available unsampled point to an existing sampled point \code{(cp.critetrion = cp.neighb)}, or as a random choice from amongst all available unsampled points within distance \eqn{zeta} of an existing sampled point \code{(cp.criterion = cp.zeta)}.
The algorithm proceeds as follows.

Let \eqn{k} be the required number of close pairs.
\itemize{
\item{Step 1.} Construct a simple inhibitory design \bold{SI}\eqn{(n - k, \delta)}.
\item{Step 2.} Sample \eqn{k} from \eqn{x_{1}, \ldots, x_{n - k}} without replacement and call this set \eqn{x_{j} : j = 1, \ldots, k}.
\item{Step 3.} For each \eqn{x_{j}: j = 1, \ldots, k}, select a close pair \eqn{x_{n-k+j}} according to the specified criterion.}

\bold{Note:} Depending on the spatial configuration of potential sampling locations and, when the selection criterion \code{cp.criterion = cp.zeta}, the specified value of \eqn{zeta}, it is possible that one or more of the selected points  \eqn{x_{j}} in Step 2 will not have an eligible ``close pair''. In this case, the algorithm will try  find an alternative \eqn{x_{j}} and report a warning if it fails to do so.
}
\note{
If \code{'delta'} is set to 0, a completely random sample is generated. In this case, \emph{'close pairs'} are not permitted and \code{'zeta'} becomes trivial.
}
\examples{
library("sf")
set.seed(1234)
x <- 0.015+0.03*(1:33)
xall <- rep(x,33)
yall <- c(t(matrix(xall,33,33)))
xy <- cbind(xall,yall)+matrix(-0.0075+0.015*runif(33*33*2),33*33,2)


# Convert to SF object
xy <- xy \%>\%
  as.data.frame \%>\%
  sf::st_as_sf(coords = c(1,2))


# Plot the points
plot(st_geometry(xy),pch=19,cex=0.25,xlab="longitude",ylab="latitude",
     cex.lab=1,cex.axis=1,cex.main=1, axes = TRUE)


# Generate spatially random sample
set.seed(15892)
xy.sample1 <- xy[sample(1:dim(xy)[1],50,replace=FALSE),]
plot(xy.sample1, pch = 19, col = 'black', add = TRUE)


set.seed(15892)
xy.sample2 <- discrete.inhibit.sample(obj=xy,size = 100,
                                      delta = 0.08,plotit = TRUE)
plot(st_geometry(xy),pch=19, cex = 0.25, col="black", add = TRUE)


# Generate spatially inhibitory sample
# with close pairs (cp.zeta criterion):
set.seed(15892)
xy.sample3 <- discrete.inhibit.sample(obj=xy, size = 100,delta = 0.065,
                                     k = 25,cp.criterion = "cp.zeta",
                                     zeta = 0.025, plotit = TRUE)
plot(st_geometry(xy),pch=19, cex = 0.25, col="black", add = TRUE)


# Generate spatially inhibitory sample
# with close pairs (cp.neighb criterion):
set.seed(15892)
xy.sample4 <- discrete.inhibit.sample(obj=xy,size = 100,
                                      delta = 0.065, k = 25,cp.criterion = "cp.neighb",
                                      plotit = TRUE)
plot(st_geometry(xy),pch=19, cex = 0.25, col="black", add = TRUE)


# Generate spatially inhibitory sample
# with close pairs (cp.zeta criterion):
set.seed(15892)
xy.sample5 <- discrete.inhibit.sample(obj=xy,size = 100,
                                      delta = 0.065, cp.criterion = "cp.zeta",
                                      zeta = 0.025, delta.fix = TRUE,
                                      k = 25, plotit = TRUE)
plot(st_geometry(xy),pch=19, cex = 0.25, col="black", add = TRUE)


# Generate simple inhibitory sample from a regular grid
library("PrevMap")
data("sim.data")
set.seed(15892)
xy.sample6 <- discrete.inhibit.sample(obj = sim.data,
                                      size = 50, delta = 0.08,plotit = TRUE)
plot(st_geometry(sim.data),pch=19,col="black", cex = 0.25, add = TRUE)


# Generate inhibitory plus close pairs sample from a regular grid
set.seed(15892)
xy.sample7 <- discrete.inhibit.sample(obj = sim.data,
                                      cp.criterion = "cp.neighb", size = 50,
                                      delta = 0.1, k = 5, plotit =TRUE)
plot(st_geometry(sim.data),pch=19,col="black", cex = 0.25, add = TRUE)
}
\references{
Chipeta  M G, Terlouw D J, Phiri K S and Diggle P J. (2016). Inhibitory geostatistical designs for spatial prediction taking account of uncertain covariance structure, \emph{Enviromentrics}, pp. 1-11.

Diggle P J. (2014). \emph{Statistical Analysis of Spatial and Spatio-Temporal Point Patterns.} 3rd ed., Boca Raton: CRC Press

Diggle P J and Lophaven S. (2006). Bayesian geostatistical design, \emph{Scandinavian Journal of Statistics} \bold{33}(1) pp. 53 - 64.
}
\author{
Michael G. Chipeta \email{mchipeta@mlw.mw}

Peter J. Diggle \email{p.diggle@lancaster.ac.uk}
}
