#' Data Generator
#'
#' @description Generating data with a given model = changepoint relative positions + parameters + type of cost + (standard deviation + gamma decay)
#' @param n number of data points to generate
#' @param changepoints vector of position of the changepoints in (0,1] (last element is always 1).
#' @param parameters vector of means for the consecutive segments (same length as changepoints)
#' @param type a string defining the cost model to use: "mean", "variance", "poisson", "exp", "negbin"
#' @param sigma a positive number = the standard deviation of the data
#' @param gamma a number between 0 and 1 : the coefficient of the exponential decay (by default = 1 for piecewise constant signals)
#' @param size parameter of the rnbinom function
#' @return a vector of size n generated by the chosen model
#' @examples
#' dataGenerator(100, c(0.3, 0.6, 1), c(1, 2, 3))
dataGenerator <- function(n, changepoints, parameters, type = "mean", sigma = 1, gamma = 1, size = 10)
{
  ### STOP ###
  if(is.numeric(n) == FALSE || n%%1 != 0){stop('n is not an integer.')}
  if(n < 2){stop('n must be a >= 1')}

  if(is.numeric(changepoints) == FALSE){stop('changepoints is not a numeric vector')}
  if(!all(changepoints <= 1) || !all(changepoints > 0)){stop('changepoints has values outside (0,1]')}
  if(changepoints[length(changepoints)] != 1){stop('the last value in changepoints vector is not 1')}
  if(length(changepoints) > 1)
  {
    for(i in 1:(length(changepoints)-1)){if(changepoints[i+1] <= changepoints[i]){stop('changepoints not an increasing vector')}}
  }

  allowed.types <- c("mean", "variance", "poisson", "exp", "negbin")
  if(!type %in% allowed.types){stop('type must be one of: ', paste(allowed.types, collapse=", "))}

  if(sigma < 0){stop('sigma is a nonnegative value')}
  if(gamma > 1 || gamma <= 0){stop('gamma is not between 0 and 1 (0 excluded)')}

  ### ###

  SegLength <- diff(c(0, floor(changepoints*n)))
  vectData <- NULL

  if(type == "mean")
  {
    if(gamma != 1)
    {
      decay <- NULL
      for(i in 1:length(SegLength)){decay <- c(decay, cumprod(rep(gamma,SegLength[i])))}
      vectData <- rep(parameters, SegLength)*decay + rnorm(n, 0, sigma)
    }
    else
    {
      vectData <- rep(parameters, SegLength) + rnorm(n, 0, sigma)
    }
  }

  if(type == "variance"){vectData <- rnorm(n, 0, rep(sqrt(parameters), SegLength))}
  if(type == "poisson"){vectData <- rpois(n, rep(parameters, SegLength))}
  if(type == "exp"){vectData <- rexp(n, rep(parameters, SegLength))}
  if(type == "negbin")
  {
    if(any(parameters > 1) || any(parameters <= 0)){stop('parameters for negbin type should be positive probabilities')}
    vectData <- rnbinom(n, size = size, prob = rep(parameters, SegLength))
  }

  return(vectData)
}

#' sdDiff
#'
#' @description Three methods to estimate the standard deviation in a time series for change-in-mean problem
#' @param x vector of datapoint
#' @param method Three available methods: "HALL", "MAD" and "SD"
#' @return a value equal to the estimated standard deviation
#' @examples
#' data <- dataGenerator(100, c(0.3, 0.6, 1), c(1, 2, 3), sigma = 2)
#' sdDiff(data)
#'
sdDiff <- function(x, method = 'HALL')
{
  #if(is.numeric(x) == FALSE || length(x) < 2){stop('x is not a numeric vector of length > 1')}
  if(method == "HALL")
  {
    n = length(x)
    wei <- c(0.1942, 0.2809, 0.3832, -0.8582)
    mat <- wei %*% t(x)
    mat[2, -n] = mat[2, -1]
    mat[3, -c(n-1, n)] = mat[3, -c(1, 2)]
    mat[4, -c(n-2, n-1, n)] = mat[4, -c(1, 2, 3)]
    return(sqrt(sum(apply(mat[, -c(n-2, n-1, n)], 2, sum)^2) / (n-3)))
  }
  if(method == "MAD")
  {
    return(mad(diff(x)/sqrt(2)))
  }
  if(method == "SD")
  {
    return(sd(diff(x)/sqrt(2)))
  }
  return(NULL)
}


###############################################
# invisible function for the user
getDerivativePenalty <- function(D, n, c1 = 2, c2 = 5)
{
  return(c1*log(n) - c1*(log(D) + 1) + c2)
}
