% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gpb.Booster.R
\name{predict.gpb.Booster}
\alias{predict.gpb.Booster}
\title{Prediction function for \code{gpb.Booster} objects}
\usage{
\method{predict}{gpb.Booster}(object, data, start_iteration = NULL,
  num_iteration = NULL, pred_latent = FALSE, predleaf = FALSE,
  predcontrib = FALSE, header = FALSE, reshape = FALSE,
  group_data_pred = NULL, group_rand_coef_data_pred = NULL,
  gp_coords_pred = NULL, gp_rand_coef_data_pred = NULL,
  cluster_ids_pred = NULL, vecchia_pred_type = NULL,
  num_neighbors_pred = -1, predict_cov_mat = FALSE, predict_var = FALSE,
  ignore_gp_model = FALSE, rawscore = NULL, ...)
}
\arguments{
\item{object}{Object of class \code{gpb.Booster}}

\item{data}{a \code{matrix} object, a \code{dgCMatrix} object or a character representing a filename}

\item{start_iteration}{int or NULL, optional (default=NULL)
Start index of the iteration to predict.
If NULL or <= 0, starts from the first iteration.}

\item{num_iteration}{int or NULL, optional (default=NULL)
Limit number of iterations in the prediction.
If NULL, if the best iteration exists and start_iteration is NULL or <= 0, the
best iteration is used; otherwise, all iterations from start_iteration are used.
If <= 0, all iterations from start_iteration are used (no limits).}

\item{pred_latent}{If TRUE latent variables, both fixed effects (tree-ensemble) 
and random effects (\code{gp_model}) are predicted. Otherwise, the response variable 
(label) is predicted. Depending on how the argument 'pred_latent' is set,
different values are returned from this function; see the 'Value' section for more details. 
If there is no \code{gp_model}, this argument corresponds to 'raw_score' in LightGBM.}

\item{predleaf}{whether predict leaf index instead.}

\item{predcontrib}{return per-feature contributions for each record.}

\item{header}{only used for prediction for text file. True if text file has header}

\item{reshape}{whether to reshape the vector of predictions to a matrix form when there are several
prediction outputs per case.}

\item{group_data_pred}{A \code{vector} or \code{matrix} with elements being group levels 
for which predictions are made (if there are grouped random effects in the \code{GPModel})}

\item{group_rand_coef_data_pred}{A \code{vector} or \code{matrix} with covariate data 
for grouped random coefficients (if there are some in the \code{GPModel})}

\item{gp_coords_pred}{A \code{matrix} with prediction coordinates (=features) for 
Gaussian process (if there is a GP in the \code{GPModel})}

\item{gp_rand_coef_data_pred}{A \code{vector} or \code{matrix} with covariate data for 
Gaussian process random coefficients (if there are some in the \code{GPModel})}

\item{cluster_ids_pred}{A \code{vector} with elements indicating the realizations of 
random effects / Gaussian processes for which predictions are made 
(set to NULL if you have not specified this when creating the \code{GPModel})}

\item{vecchia_pred_type}{A \code{string} specifying the type of Vecchia approximation used for making predictions. 
"order_obs_first_cond_obs_only" = observed data is ordered first and the neighbors are only observed points, 
"order_obs_first_cond_all" = observed data is ordered first and the neighbors are selected among all points 
(observed + predicted), "order_pred_first" = predicted data is ordered first for making predictions, 
"latent_order_obs_first_cond_obs_only" = Vecchia approximation for the latent process and observed data is 
ordered first and neighbors are only observed points, "latent_order_obs_first_cond_all" = Vecchia approximation 
for the latent process and observed data is ordered first and neighbors are selected among all points}

\item{num_neighbors_pred}{an \code{integer} specifying the number of neighbors for the Vecchia approximation 
for making predictions}

\item{predict_cov_mat}{A \code{boolean}. If TRUE, the (posterior) 
predictive covariance is calculated in addition to the (posterior) predictive mean}

\item{predict_var}{A \code{boolean}. If TRUE, the (posterior) 
predictive variances are calculated}

\item{ignore_gp_model}{A \code{boolean}. If TRUE, predictions are only made for the tree ensemble part
and the \code{gp_model} is ignored}

\item{rawscore}{This is deprecated, use the renamed equivalent argument 
\code{pred_latent} instead}

\item{...}{Additional named arguments passed to the \code{predict()} method of
the \code{gpb.Booster} object passed to \code{object}.}
}
\value{
either a list with vectors or a single vector / matrix depending on 
whether there is a \code{gp_model} or not
  If there is a \code{gp_model}, the result dict contains the following entries.
    1. If \code{pred_latent} is TRUE, the dict contains the following 3 entries:
      - result["fixed_effect"] are the predictions from the tree-ensemble.
      - result["random_effect_mean"] are the predicted means of the \code{gp_model}.
      - result["random_effect_cov"] are the predicted covariances or variances of the \code{gp_model}
  (only if 'predict_var' or 'predict_cov' is TRUE).
    2. If \code{pred_latent} is FALSE, the dict contains the following 2 entries:
      - result["response_mean"] are the predicted means of the response variable (Label) taking into account
    both the fixed effects (tree-ensemble) and the random effects (\code{gp_model})
      - result["response_var"] are the predicted  covariances or variances of the response variable
  (only if 'predict_var' or 'predict_cov' is TRUE)
  If there is no \code{gp_model} or \code{predcontrib} or \code{ignore_gp_model} 
    are TRUE, the result contains predictions from the tree-booster only.
}
\description{
Prediction function for \code{gpb.Booster} objects
}
\examples{

# See https://github.com/fabsig/GPBoost/tree/master/R-package for more examples

library(gpboost)
data(GPBoost_data, package = "gpboost")

#--------------------Combine tree-boosting and grouped random effects model----------------
# Create random effects model
gp_model <- GPModel(group_data = group_data[,1], likelihood = "gaussian")
# The default optimizer for covariance parameters (hyperparameters) is 
# Nesterov-accelerated gradient descent.
# This can be changed to, e.g., Nelder-Mead as follows:
# re_params <- list(optimizer_cov = "nelder_mead")
# gp_model$set_optim_params(params=re_params)
# Use trace = TRUE to monitor convergence:
# re_params <- list(trace = TRUE)
# gp_model$set_optim_params(params=re_params)

# Train model
bst <- gpboost(data = X, label = y, gp_model = gp_model, nrounds = 16,
               learning_rate = 0.05, max_depth = 6, min_data_in_leaf = 5,
               objective = "regression_l2", verbose = 0)
# Estimated random effects model
summary(gp_model)

# Make predictions
# Predict latent variables
pred <- predict(bst, data = X_test, group_data_pred = group_data_test[,1],
                predict_var = TRUE, pred_latent = TRUE)
pred$random_effect_mean # Predicted latent random effects mean
pred$random_effect_cov # Predicted random effects variances
pred$fixed_effect # Predicted fixed effects from tree ensemble
# Predict response variable
pred_resp <- predict(bst, data = X_test, group_data_pred = group_data_test[,1],
                     predict_var = TRUE, pred_latent = FALSE)
pred_resp$response_mean # Predicted response mean
# For Gaussian data: pred$random_effect_mean + pred$fixed_effect = pred_resp$response_mean
pred$random_effect_mean + pred$fixed_effect - pred_resp$response_mean

\donttest{
#--------------------Combine tree-boosting and Gaussian process model----------------
# Create Gaussian process model
gp_model <- GPModel(gp_coords = coords, cov_function = "exponential",
                    likelihood = "gaussian")
# Train model
bst <- gpboost(data = X, label = y, gp_model = gp_model, nrounds = 8,
               learning_rate = 0.1, max_depth = 6, min_data_in_leaf = 5,
               objective = "regression_l2", verbose = 0)
# Estimated random effects model
summary(gp_model)
# Make predictions
pred <- predict(bst, data = X_test, gp_coords_pred = coords_test,
                predict_var = TRUE, pred_latent = TRUE)
pred$random_effect_mean # Predicted latent random effects mean
pred$random_effect_cov # Predicted random effects variances
pred$fixed_effect # Predicted fixed effects from tree ensemble
# Predict response variable
pred_resp <- predict(bst, data = X_test, gp_coords_pred = coords_test,
                     predict_var = TRUE, pred_latent = FALSE)
pred_resp$response_mean # Predicted response mean
}
}
\author{
Fabio Sigrist, authors of the LightGBM R package
}
