% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/threshold.R
\docType{methods}
\name{threshold}
\alias{threshold}
\alias{threshold,mpg-method}
\title{Produce a minimum planar graph (MPG) at multiple scales}
\usage{
threshold(x, ...)

\S4method{threshold}{mpg}(x, weight = "lcpPerimWeight", nThresh = NULL,
  doThresh = NULL, ...)
}
\arguments{
\item{x}{A \code{mpg} object produced by \code{\link{MPG}}.}

\item{...}{Additional arguments (not used).}

\item{weight}{A string giving the link weight or attribute to use for threshold.
\code{"lcpPerimWeight"} uses the accumulated resistance or least-cost path
distance from the perimeters of patches as the link weight.
\code{"eucPerimWeight"} use the Euclidean distance from the
perimeters of patches as the link weight.}

\item{nThresh}{Optional. An integer giving the number of thresholds (or scales)
at which to create GOC models.
Thresholds are selected to produce a maximum number of unique
grains (i.e., models).
\code{nThresh} thresholds are also approximately evenly spread
between 0 and the threshold at which all patches or focal points
on the landscape are connected.
This is a simple way to get a representative subset of all
possible GOC models.
Provide either \code{nThresh} or \code{doThresh} not both.}

\item{doThresh}{Optional. A vector giving the link thresholds at which to create GOC models.
Use \code{link{threshold}} to identify thresholds of interest.
Provide either \code{nThresh} or \code{doThresh} not both.}
}
\value{
A list object with the following elements:

\describe{
  \item{\code{summary}}{summarizes the thresholded graphs generated and their properties;}
  \item{\code{th}}{a list of length \code{nThresh} or \code{length(doThresh)}
  giving the thresholded graph (class \code{igraph}) at each threshold.}
}
}
\description{
Perform a scalar analysis of a minimum planar graph (MPG) by building the
graph at a series of link thresholds.
As the threshold value increases more nodes in the graph become connected,
forming increasingly fewer components, until the graph becomes connected (e.g., Brooks, 2003).
N.B. Grains of connectivity (GOC) done by \code{\link{GOC}} is also a scalar
analysis using Voronoi tessellations rather than patches (see Galpern \emph{et al.}, 2012).
}
\note{
See \code{\link{MPG}} for warning related to areal measurements.
}
\examples{
## Load raster landscape
tiny <- raster::raster(system.file("extdata/tiny.asc", package = "grainscape"))

## Create a resistance surface from a raster using an is-becomes reclassification
tinyCost <- raster::reclassify(tiny, rcl = cbind(c(1, 2, 3, 4), c(1, 5, 10, 12)))

## Produce a patch-based MPG where patches are resistance features=1
tinyPatchMPG <- MPG(cost = tinyCost, patch = tinyCost == 1)

## Threshold this graph at a representative subset of 10 thresholds
tinyThresh <- threshold(tinyPatchMPG, nThresh = 10)

## Examine the properties of one of these threshold graphs
print(tinyThresh$th[[7]], vertex = TRUE, edge = TRUE)
}
\references{
Brooks, C.P. (2003) A scalar analysis of landscape connectivity. Oikos 102:433-439.

Fall, A., M.-J. Fortin, M. Manseau, D. O'Brien. (2007) Spatial graphs:
Principles and applications for habitat connectivity. Ecosystems 10:448:461.

Galpern, P., M. Manseau. (2013a) Finding the functional grain: comparing methods
for scaling resistance surfaces. Landscape Ecology 28:1269-1291.

Galpern, P., M. Manseau. (2013b) Modelling the influence of landscape connectivity
on animal distribution: a functional grain approach. Ecography 36:1004-1016.

Galpern, P., M. Manseau, A. Fall. (2011) Patch-based graphs of landscape connectivity:
a guide to construction, analysis, and application for conservation.
Biological Conservation 144:44-55.

Galpern, P., M. Manseau, P.J. Wilson. (2012) Grains of connectivity: analysis
at multiple spatial scales in landscape genetics. Molecular Ecology 21:3996-4009.
}
\seealso{
\code{\link{MPG}}
}
\author{
Paul Galpern and Alex Chubaty
}
