% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_mc_oral_equiv.R
\name{calc_mc_oral_equiv}
\alias{calc_mc_oral_equiv}
\title{Calculate Monte Carlo Oral Equivalent Dose}
\usage{
calc_mc_oral_equiv(
  conc,
  chem.name = NULL,
  chem.cas = NULL,
  dtxsid = NULL,
  parameters = NULL,
  which.quantile = 0.95,
  species = "Human",
  input.units = "uM",
  output.units = "mgpkgpday",
  suppress.messages = FALSE,
  return.samples = FALSE,
  restrictive.clearance = TRUE,
  bioactive.free.invivo = FALSE,
  tissue = NULL,
  concentration = "plasma",
  IVIVE = NULL,
  model = "3compartmentss",
  Caco2.options = list(),
  calc.analytic.css.arg.list = list(),
  ...
)
}
\arguments{
\item{conc}{Bioactive in vitro concentration in units of uM.}

\item{chem.name}{Either the chemical name or the CAS number must be
specified.}

\item{chem.cas}{Either the CAS number or the chemical name must be
specified.}

\item{dtxsid}{EPA's 'DSSTox Structure ID (\url{https://comptox.epa.gov/dashboard})
the chemical must be identified by either CAS, name, or DTXSIDs}

\item{parameters}{Parameters from the appropriate parameterization function
for the model indicated by argument model}

\item{which.quantile}{Which quantile from Monte Carlo steady-state
simulation (\code{\link{calc_mc_css}}) is requested. Can be a vector. Note that 95th
concentration quantile is the same population as the 5th dose quantile.}

\item{species}{Species desired (either "Rat", "Rabbit", "Dog", "Mouse", or
default "Human").}

\item{input.units}{Units of given concentration, default of uM but can also
be mg/L.}

\item{output.units}{Units of dose, default of 'mgpkgpday' for mg/kg BW/ day or
'umolpkgpday' for umol/ kg BW/ day.}

\item{suppress.messages}{Suppress text messages.}

\item{return.samples}{Whether or not to return the vector containing the
samples from the simulation instead of the selected quantile.}

\item{restrictive.clearance}{Protein binding not taken into account (set to
1) in liver clearance if FALSE.}

\item{bioactive.free.invivo}{If FALSE (default), then the total concentration is treated
as bioactive in vivo. If TRUE, the the unbound (free) plasma concentration is treated as 
bioactive in vivo. Only works with tissue = NULL in current implementation.}

\item{tissue}{Desired steady state tissue concentration. Default is of NULL
typically gives whole body plasma concentration.}

\item{concentration}{Desired concentration type: 'blood','tissue', or default 
'plasma'. In the case that the concentration is for plasma, selecting "blood"
will use the blood:plasma ratio to estimate blood concentration. In the case
that the argument 'tissue' specifies a particular tissue of the body, 
concentration defaults to 'tissue' -- that is, the concentration in the 
If cocentration is set to 'blood' or 'plasma' and 'tissue' specifies a
specific tissue then the value returned is for the plasma or blood in that
specific tissue.}

\item{IVIVE}{Honda et al. (2019) identified six plausible sets of
assumptions for \emph{in vitro-in vivo} extrapolation (IVIVE) assumptions.
Argument may be set to "Honda1" through "Honda6". If used, this function
overwrites the tissue, restrictive.clearance, and bioactive.free.invivo arguments.
See Details below for more information.}

\item{model}{Model used in calculation,'gas_pbtk' for the gas pbtk model, 
'pbtk' for the multiple compartment model,
'3compartment' for the three compartment model, '3compartmentss' for 
the three compartment steady state model, and '1compartment' for one 
compartment model. This only applies when httkpop=TRUE and species="Human",
otherwise '3compartmentss' is used.}

\item{Caco2.options}{A list of options to use when working with Caco2 apical to
basolateral data \code{Caco2.Pab}, default is Caco2.options = list(Caco2.default = 2,
Caco2.Fabs = TRUE, Caco2.Fgut = TRUE, overwrite.invivo = FALSE, keepit100 = FALSE). Caco2.default sets the default value for 
Caco2.Pab if Caco2.Pab is unavailable. Caco2.Fabs = TRUE uses Caco2.Pab to calculate
fabs.oral, otherwise fabs.oral = \code{Fabs}. Caco2.Fgut = TRUE uses Caco2.Pab to calculate 
fgut.oral, otherwise fgut.oral = \code{Fgut}. overwrite.invivo = TRUE overwrites Fabs and Fgut in vivo values from literature with 
Caco2 derived values if available. keepit100 = TRUE overwrites Fabs and Fgut with 1 (i.e. 100 percent) regardless of other settings.}

\item{calc.analytic.css.arg.list}{A list of options to pass to the analytic steady-state calculation function.
This includes `restrictive.clearance`, `bioactive.free.invivo`, `IVIVE`,
`wellstirred.correction`, and `adjusted.Funbound.plasma`.}

\item{...}{Additional parameters passed to \code{\link{calc_mc_css}} for httkpop and
variance of parameters.}
}
\value{
Equivalent dose in specified units, default of mg/kg BW/day.
}
\description{
This function converts a chemical plasma concetration to an oral adminstered
equivalent
dose (AED) using a concentration obtained from \code{\link{calc_mc_css}}.  
This function uses reverse dosimetry-based 
'\emph{in vitro-in vivo} extrapolation (IVIVE) 
for high throughput risk screening.  The user can input the
chemical and \emph{in vitro} bioactive concentration, select the TK
model, and then automatically predict the \emph{in vivo} AED
which would produce a body concentration equal to
the \emph{in vitro} bioactive concentration. This function relies on the
Monte Carlo method (via funcion \code{\link{create_mc_samples}} to simulate
both uncertainty and variability so that the result is a distribution of
equivalent doses, from which we provide specific quantiles
(specified by \code{which.quantile}), though the full set of predictions can
be obtained by setting \code{return.samples} to \code{TRUE}.
}
\details{
The chemical-specific steady-state concentration for a dose rate of 
1 mg/kg body weight/day can be used for in IVIVE of
a bioactive \emph{in vitro} concentration by dividing the \emph{in vitro}
concentration
by the steady-state concentration to predict a dose rate (mg/kg/day) that
would produce that concentration in plasma. Using quantiles of the 
distribution (such as the upper 95th percentile) allow incorporation of 
uncertainty and variability into IVIVE.

This approach relies on thelinearity
of the models to calculate a scaling factor to relate in vitro
concentrations (uM) with AED. The scaling factor is the
inverse of the steady-state plasma concentration (Css) predicted
for a 1 mg/kg/day exposure dose rate
where \emph{in vitro} concentration [X] and Css must be in the same
units. Note that it is typical for \emph{in vitro} concentrations to be
reported in units of uM and Css in units of mg/L, in which case
one must be converted to the other. 

Reverse Dosimetry Toxicodynamic IVIVE

\if{html}{\figure{ivive.png}{options: width="60\%" alt="Reverse Dosimetry Toxicodynamic IVIVE"}}
\if{latex}{\figure{ivive.pdf}{options: width=12cm alt="Reverse Dosimetry Toxicodynamic IVIVE"}}

Figure from Breen et al. (2021) (\doi{10.1080/17425255.2021.1935867}) shows 
reverse dosimetry toxicodynamic IVIVE. Equivalent external dose is
determined by solving the TK model in reverse by deriving the external dose
(that is, TK model input) that produces a specified internal concentration 
(that is, TK model output). Reverse dosimetry and IVIVE using HTTK relies on 
the linearity of the models. We calculate a scaling factor to relate 
\emph{in vitro} 
concentrations (uM) to AEDs. The scaling 
factor is the inverse of the Css 
predicted for a 1 mg/kg/day exposure dose rate. We use Monte Carlo to 
simulate variability and propagate uncertainty; for example, to calculate an 
upper 95th percentile Css,95 for individuals who get higher plasma 
concentrations from the same exposure.

The Monte Carlo methods used here were recently updated and described by
Breen et al. (submitted).

All arguments after httkpop only apply if httkpop is set to TRUE and species
to "Human".

When species is specified as rabbit, dog, or mouse, the function uses the
appropriate physiological data(volumes and flows) but substitutes human
fraction unbound, partition coefficients, and intrinsic hepatic clearance.

Tissue concentrations are calculated for the pbtk model with oral infusion
dosing.  All tissues other than gut, liver, and lung are the product of the
steady state plasma concentration and the tissue to plasma partition
coefficient.

The six sets of plausible IVIVE
assumptions identified by Honda et al. (2019) 
(\doi{10.1371/journal.pone.0217564}) 
are: \tabular{lrrrr}{
\tab \emph{in vivo} Conc. \tab Metabolic Clearance \tab Bioactive Chemical
Conc. \tab TK Statistic Used* \cr Honda1 \tab Veinous (Plasma) \tab
Restrictive \tab Free \tab Mean Conc. \cr Honda2 \tab Veinous \tab
Restrictive \tab Free \tab Max Conc. \cr Honda3 \tab Veinous \tab
Non-restrictive \tab Total \tab Mean Conc. \cr Honda4 \tab Veinous \tab
Non-restrictive \tab Total \tab Max Conc. \cr Honda5 \tab Target Tissue \tab
Non-restrictive \tab Total \tab Mean Conc. \cr Honda6 \tab Target Tissue
\tab Non-restrictive \tab Total \tab Max Conc. \cr } *Assumption is
currently ignored because analytical steady-state solutions are currently
used by this function.
}
\examples{
\donttest{
# Set the number of samples (NSAMP) low for rapid testing, increase NSAMP 
# for more stable results. Default value is 1000:
NSAMP = 10

# Basic in vitro - in vivo extrapolation with httk, convert 0.5 uM in vitro
# concentration of chemical Surinabant to mg/kg/day:
set.seed(1234)
0.5/calc_mc_css(chem.name="Surinabant", samples=NSAMP, output.units="uM")

# The significant digits should give the same answer as:
set.seed(1234)
calc_mc_oral_equiv(chem.name="Surinabant",conc=0.5,samples=NSAMP)  

# Note that we use set.seed to get the same sequence of random numbers for
# the two different function calls (calc_mc_css and calc_mc_oral_equiv)

# The following example should result in an error since we do not 
# estimate tissue partitioning with '3compartmentss'. 
set.seed(1234)                        
try(calc_mc_oral_equiv(0.1, chem.cas="34256-82-1",
                       which.quantile=c(0.05,0.5,0.95),
                       samples=NSAMP,
                       tissue='brain'))
       
set.seed(1234)
calc_mc_oral_equiv(0.1,chem.cas="34256-82-1", model='pbtk',
                   samples=NSAMP,
                   which.quantile=c(0.05,0.5,0.95), tissue='brain')
 
# We can also use the Monte Carlo functions by passing a table
# where each row represents a different Monte Carlo draw of parameters:
p <- create_mc_samples(chem.cas="80-05-7")
# Use data.table for steady-state plasma concentration (Css) Monte Carlo:
calc_mc_css(parameters=p)
# Using the same table gives the same answer:
calc_mc_css(parameters=p)
# Use Css for 1 mg/kg/day for simple reverse toxicokinetics 
# in Vitro-In Vivo Extrapolation to convert 15 uM to mg/kg/day:
15/calc_mc_css(parameters=p, output.units="uM")
# Can do the same with calc_mc_oral_equiv:
calc_mc_oral_equiv(15, parameters=p)
}

}
\references{
\insertRef{ring2017identifying}{httk} 

\insertRef{wetmore2015incorporating}{httk}

\insertRef{honda2019using}{httk}

\insertRef{rowland1973clearance}{httk}
}
\seealso{
\code{\link{calc_mc_css}}

\code{\link{create_mc_samples}}
}
\author{
John Wambaugh
}
\keyword{Monte-Carlo}
\keyword{Steady-State}
