% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FeatureImp.R
\name{FeatureImp}
\alias{FeatureImp}
\title{Feature importance}
\format{\code{\link{R6Class}} object.}
\description{
\code{FeatureImp} computes feature importances for prediction models.
The importance is measured as the factor by which the model's prediction error increases when the feature is shuffled.
}
\section{Usage}{

\preformatted{
imp = FeatureImp$new(predictor, loss, compare = "ratio", n.repetitions = 5)

imp$plot()
imp$results
print(imp)
}
}

\section{Arguments}{


For FeatureImp$new():
\describe{
\item{predictor: }{(Predictor)\cr 
The object (created with Predictor$new()) holding the machine learning model and the data.}
\item{loss: }{(`character(1)` | function)\cr The loss function. Either the name of a loss (e.g. "ce" for classification or "mse") or a function. See Details for allowed losses.}
\item{compare: }{(`character(1)`)\cr Either "ratio" or "difference". 
Should importance be measured as the difference or as the ratio of original model error and model error after permutation?
Ratio: error.permutation/error.orig,   Difference: error.permutation - error.orig}
\item{n.repetitions: }{`numeric(1)`\cr How often should the shuffling of the feature be repeated?
The higher the number of repetitions the more stable and accurate the results become. 
}
\item{parallel: }{`logical(1)`\cr Should the method be executed in parallel? If TRUE, requires a cluster to be registered, see ?foreach::foreach.}
}
}

\section{Details}{


To compute the feature importance for a single feature, the model prediction loss (error) is measured before
and after shuffling the values of the feature. By shuffling the feature values, the association between the 
outcome and the feature is destroyed. The larger the increase in prediction error, the more important
the feature was. The shuffling is repeated to get more accurate results, since the permutation 
feature importance tends to be quite instable.
Read the Interpretable Machine Learning book to learn about feature importance  in detail: 
\url{https://christophm.github.io/interpretable-ml-book/feature-importance.html}


The loss function can be either specified via a string, or by handing a function to \code{FeatureImp()}.
If you want to use your own loss function it should have this signature: function(actual, predicted).
Using the string is a shortcut to using loss functions from the \code{Metrics} package. 
Only use functions that return a single performance value, not a vector.
Allowed losses are: "ce", "f1", "logLoss", "mae", "mse", "rmse", "mape", "mdae",
"msle", "percent_bias", "rae", "rmse", "rmsle", "rse", "rrse", "smape"
See \code{library(help = "Metrics")} to get a list of functions.
}

\section{Fields}{

\describe{
\item{original.error: }{(`numeric(1)`)\cr The loss of the model before perturbing features.}
\item{predictor: }{(Predictor)\cr The prediction model that was analysed.}
\item{compare: }{(`character(1)`)\cr Either "ratio" or "difference", depending on whether the importance was calculated as difference between original model error and model error after permutation or as ratio.}
\item{results: }{(data.frame)\cr data.frame with the results of the feature importance computation. One row per feature with the following columns:
                 importance.05 (5% quantile of importance values from the repetitions), 
                 importance (median importance),
                 importance.95 (95% quantile) and the permutation.error (median error over all repetitions). The distribution of the importance 
                 is also visualized as a bar in the plots, the median importance over the repetitions as a point.}
}
}

\section{Methods}{

\describe{
\item{loss(actual,predicted)}{The loss function. Can also be applied to data: \code{object$loss(actual, predicted)}}
\item{plot()}{method to plot the feature importances. See \link{plot.FeatureImp}}
\item{\code{clone()}}{[internal] method to clone the R6 object.}
\item{\code{initialize()}}{[internal] method to initialize the R6 object.}
}
}

\examples{
if (require("rpart")) {
# We train a tree on the Boston dataset:
data("Boston", package  = "MASS")
tree = rpart(medv ~ ., data = Boston)
y = Boston$medv
X = Boston[-which(names(Boston) == "medv")]
mod = Predictor$new(tree, data = X, y = y)


# Compute feature importances as the performance drop in mean absolute error
imp = FeatureImp$new(mod, loss = "mae")

# Plot the results directly
plot(imp)


# Since the result is a ggplot object, you can extend it: 
if (require("ggplot2")) {
  plot(imp) + theme_bw()
  # If you want to do your own thing, just extract the data: 
  imp.dat = imp$results
  head(imp.dat)
  ggplot(imp.dat, aes(x = feature, y = importance)) + geom_point() + 
  theme_bw()
}

# We can also look at the difference in model error instead of the ratio
imp = FeatureImp$new(mod, loss = "mae", compare = "difference")

# Plot the results directly
plot(imp)


# FeatureImp also works with multiclass classification. 
# In this case, the importance measurement regards all classes
tree = rpart(Species ~ ., data= iris)
X = iris[-which(names(iris) == "Species")]
y = iris$Species
mod = Predictor$new(tree, data = X, y = y, type = "prob") 

# For some models we have to specify additional arguments for the predict function
imp = FeatureImp$new(mod, loss = "ce")
plot(imp)

# For multiclass classification models, you can choose to only compute performance for one class. 
# Make sure to adapt y
mod = Predictor$new(tree, data = X, y = y == "virginica", 
 type = "prob", class = "virginica") 
imp = FeatureImp$new(mod, loss = "ce")
plot(imp)
}
}
\references{
Fisher, A., Rudin, C., and Dominici, F. (2018). Model Class Reliance: Variable Importance Measures for any Machine Learning Model Class, from the "Rashomon" Perspective. Retrieved from http://arxiv.org/abs/1801.01489
}
