% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ciregic_aipw.R
\name{ciregic_aipw}
\alias{ciregic_aipw}
\title{Competing Risks Regression with Interval-Censored Data and Missing Cause of Failure}
\usage{
ciregic_aipw(
  formula,
  aux = NULL,
  data,
  alpha,
  k = 1,
  do.par,
  nboot,
  w.cores = NULL,
  ...
)
}
\arguments{
\item{formula}{a formula object relating the survival object \code{Surv2(v, u, event)} to a set of covariates}

\item{aux}{auxiliary variable(s) that may be associated with the missingness and the outcome of interest}

\item{data}{a data frame that includes the variables named in the formula argument}

\item{alpha}{\eqn{\alpha = (\alpha1, \alpha2)} contains parameters that define the link functions from class of generalized odds-rate transformation models. The components \eqn{\alpha1} and \eqn{\alpha2} should both be \eqn{\ge 0}. If \eqn{\alpha1 = 0}, the user assumes the proportional subdistribution hazards model or the Fine-Gray model for the event type 1. If \eqn{\alpha2 = 1}, the user assumes the proportional odds model for the event type 2.}

\item{k}{a parameter that controls the number of knots in the B-spline with \eqn{0.5 \le }\code{k}\eqn{ \le 1}}

\item{do.par}{an option to use parallel computing for bootstrap. If \code{do.par = TRUE}, parallel computing will be used during the bootstrap estimation of the variance-covariance matrix for the regression parameter estimates.}

\item{nboot}{a number of bootstrap samples for estimating variances and covariances of the estimated regression coefficients. If \code{nboot = 0}, the function \code{ciregic_aipw} does not perform bootstrap estimation of the variance-covariance matrix of the regression parameter estimates and returns \code{NA} in the place of the estimated variance-covariance matrix of the regression parameter estimates.}

\item{w.cores}{a number of cores that are assigned (the default is \code{NULL})}

\item{...}{further arguments}
}
\value{
The function \code{ciregic_aipw} provides an object of class \code{ciregic_aipw} with components:
\item{varnames}{a vector containing variable names}
\item{varnames.aux}{a vector containing auxiliary variable names}
\item{coefficients}{a vector of the regression coefficient estimates}
\item{gamma}{a vector of the estimated coefficients for the B-splines}
\item{vcov}{a variance-covariance matrix of the estimated regression coefficients}
\item{alpha}{a vector of the link function parameters}
\item{loglikelihood}{a loglikelihood of the fitted model}
\item{convergence}{an indicator of convegence}
\item{tms}{a vector of the minimum and maximum observation times}
\item{Bv}{a list containing the B-splines basis functions evaluated at \code{v}}
\item{numboot}{a number of converged bootstrap}
\item{notconverged}{a list of number of bootstrap samples that did not converge}
\item{call}{a matched call}
}
\description{
The function \code{ciregic_aipw} performs semiparametric regression on cumulative incidence function with interval-censored competing risks data in the presence of missing cause of failure. It fits the proportional subdistribution hazards model (Fine-Gray model), the proportional odds model, and other models that belong to the class of semiparametric generalized odds rate transformation models. The estimates have double robustness property, which means that the estimators are consistent even if either the model for the probability of missingness or the model for the probability of the cause of failure is misspecified under the missing at random assumption.
}
\details{
The formula for the model has the form of \code{response ~ predictors}. The response in the formula is a \code{Surv2(v, u, event)} object where \code{v} is the last observation time prior to the event, \code{u} is the first observation time after the event, and \code{event} is the event or censoring indicator. \code{event} should include 0, 1 or 2, denoting right-censoring, event type 1 and 2, respectively. If \code{event=0} (i.e. right-censored observation) then \code{u} is not included in any calculation as it corresponds to \eqn{\infty}. The user can provide any value in \code{u} for the right-censored cases, even \code{NA}. The function \code{ciregic_aipw} fits models that belong to the class of generalized odds rate transformation models which includes the proportional subdistribution hazards or the Fine-Gray model and the proportional odds model. The parameter \eqn{\alpha=(\alpha1, \alpha2)} defines the link function/model to be fitted for event 1 and 2, respectively. A value of \code{0} corresponds to the Fine-Gray model and a value of \code{1} corresponds to the proportional odds model. For example, if \eqn{\alpha=(0,1)} then the function \code{ciregic_aipw} fits the Fine-Gray model for the event type 1 and the proportional odds model for the event type 2.
}
\examples{
\dontrun{
## Set seed in order to have reproducibility of the bootstrap standard error estimate
set.seed(1234)

## Estimation of regression parameters only. No bootstrap variance estimation.
## with 'simdata_aipw'
data(simdata_aipw)
fit_aipw <- ciregic_aipw(formula = Surv2(v = v, u = u, event = c) ~ z1 + z2, aux = a,
                         data = simdata_aipw, alpha = c(1, 1), nboot = 0,
                         do.par = FALSE)
fit_aipw
## Bootstrap variance estimation based on 50 replications
fit_aipw <- ciregic_aipw(formula = Surv2(v = v, u = u, event = c) ~ z1 + z2, aux = a,
                         data = simdata_aipw, alpha = c(1, 1), k = 1, nboot = 50,
                         do.par = FALSE)
}
## Note that the user can use parallel computing to decrease
## the computation time of the bootstrap variance-covariance
## estimation (e.g. nboot = 50)

## Summarize semiparametric regression model
summary(fit_aipw)

## Predict and draw plot the cumulative incidence function evaluated at z1 = 1 and z2 = 0.5
t <- seq(from = 0, to = 2.8, by = 2.8 / 99)
pred <- predict(object = fit_aipw, covp = c(1, 0.5), times = t)
pred
plot(pred$t, pred$cif1, type = "l", ylim = c(0, 1))
points(pred$t, pred$cif2, type = "l", col = 2)

}
\references{
{Bakoyannis, G., Yu, M., and Yiannoutsos C. T. (2017). Semiparametric regression on cumulative incidence function with interval-censored competing risks data. \emph{Statistics in Medicine}, \strong{36}:3683-3707.}

{Fine, J. P. and Gray, R. J. (1999). A proportional hazards model for the subdistribution of a competing risk. \emph{Journal of the American Statistical Association}, \strong{94}:496-509.}
}
\seealso{
\code{\link[intccr]{summary.ciregic_aipw}} for the summarized results and \code{\link[intccr]{predict.ciregic_aipw}} for value of the predicted cumulative incidence functions. \code{coef} and \code{vcov} are the generic functions. dataprep function for reshaping data from a long format to a suitable format to be used in the function \code{ciregic_aipw}.
}
\author{
Jun Park, \email{jp84 at iu dot edu}

Giorgos Bakoyannis, \email{gbakogia at iu dot edu}
}
\keyword{ciregic_aipw}
