#' Generic summary function for dpmmeta object in jarbes
#' @param object The object generated by the dmpmeta function.
#'
#' @param digits The number of significant digits printed. The default value is 3.
#' @param ... \dots
#'
#' @export
summary.dpmeta = function(object, digits = 3, ...) {

  bugs.output = object$BUGSoutput
  bugs.summary = bugs.output$summary

  summary.m = list()

  # Model specifications ....
  #
  model.spec = list()
  model.spec$link = "Normal approximation"

  # Hyper-priors parameters............................................
  model.spec$mean.mu.0           = object$prior$mean.mu.0
  model.spec$sd.mu.0             = object$prior$sd.mu.0
  model.spec$scale.sigma.between = object$prior$scale.sigma.between
  model.spec$df.scale.between    = object$prior$df.scale.between
  model.spec$alpha.0             = object$prior$alpha.0
  model.spec$K                   = object$prior$K
  #

  summary.m$model.specification = model.spec

  # Posterior of the model parameters
  #
  # The list of parameters will include more complex models, e.g. estimation of
  # the parameters nu from the Beta ...
  #
  summary.m$summary.par = bugs.summary[c("mu.0",
                                         "sd.0",
                                         "alpha",
                                         "mu.k",
                                          "p"),]
  K = model.spec$K

  row.names(summary.m$summary.par) = c("Mean (Pooled mean)",
                                       "Tau (between studies sd)",
                                       "Intensity parameter",
                                       rep("Cluster mean",K),
                                       rep("Cluster tau", K),
                                       rep("Cluster probability",K))

    # DIC
  summary.m$DIC = bugs.output$DIC
  summary.m$pD = bugs.output$pD

  # MCMC setup ...
  mcmc.setup = list()
  mcmc.setup$n.chains = bugs.output$n.chains
  mcmc.setup$n.iter = bugs.output$n.iter
  mcmc.setup$n.burnin = bugs.output$n.burnin
  summary.m$mcmc.setup = mcmc.setup

  class(summary.m) = "summary.dpmeta"

  print(summary.m, digits, ...)

}

#'
#' @export

print.summary.dpmeta = function(x, digits, ...) {
  cat('Model specifications:\n')
  model.spec = x$model.specification

  cat(paste('  Link function: ', model.spec$link, sep = ''))
  cat('\n')
  cat('  Hyper-priors parameters: \n')
  cat(paste('  Prior for mu.0: Normal', '[', model.spec$mean.mu.0,', ' ,model.spec$sd.mu.0^2,']', sep = ''))
  cat('\n')
  cat(paste('  Prior for 1/tau^2: Scale.Gamma', '[', model.spec$scale.sigma.between,', ' ,
            model.spec$df.scale.between,']', sep = ''))
  cat('\n')
  cat(paste('  Prior for alpha: Gamma', '[', model.spec$alpha.0,', ' ,
                                             model.spec$alpha.0,']', sep = ''))

  cat('Posterior distributions: \n')
  print(round(x$summary.par, digits))

  cat('\n-------------------\n')


  mcmc = x$mcmc.setup
  cat(paste('MCMC setup (fit using jags): ', mcmc$n.chains, ' chains, each with ', mcmc$n.iter, ' iterations (first ', mcmc$n.burnin, ' discarded)', sep = ''))
  cat('\n')
  cat(paste('DIC: ', round(x$DIC, digits), sep = ''))
  cat('\n')
  cat(paste('pD: ', round(x$pD, digits), sep = ''))
  cat('\n')
}

