% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matrixcreation.R
\name{rlefko2}
\alias{rlefko2}
\title{Create Raw Ahistorical Matrix Projection Model}
\usage{
rlefko2(
  data,
  stageframe,
  year = "all",
  pop = NA,
  patch = NA,
  censor = FALSE,
  stages = NA,
  alive = c("alive3", "alive2"),
  size = c("sizea3", "sizea2"),
  sizeb = c(NA, NA),
  sizec = c(NA, NA),
  repst = c("repstatus3", "repstatus2"),
  matst = c("matstatus3", "matstatus2"),
  fec = c("feca3", "feca2"),
  supplement = NULL,
  repmatrix = NULL,
  overwrite = NULL,
  yearcol = NA,
  popcol = NA,
  patchcol = NA,
  indivcol = NA,
  censorcol = NA,
  censorkeep = 0,
  reduce = FALSE
)
}
\arguments{
\item{data}{A vertical demographic data frame, with variables corresponding 
to the naming conventions in \code{\link{verticalize3}()}.}

\item{stageframe}{A stageframe object that includes information on the size,
observation status, propagule status, immaturity status, and maturity status
of each ahistorical stage. Should also incorporate bin widths if size is
continuous.}

\item{year}{A variable corresponding to observation occasion, or a set
of such values, given in values associated with the \code{year} term used in
vital rate model development. Can also equal \code{all}, in which case
matrices will be estimated for all occasion times. Defaults to \code{all}.}

\item{pop}{A variable designating which populations will have matrices
estimated. Should be set to specific population names, or to \code{all} if
all populations should have matrices estimated.}

\item{patch}{A variable designating which patches or subpopulations will have
matrices estimated. Should be set to specific patch names, or to \code{all}
if matrices should be estimated for all patches. Defaults to \code{all}.}

\item{censor}{If TRUE, then data will be removed according to the variable
set in \code{censorcol}, such that only data with censor values equal to 1
will remain. Defaults to FALSE.}

\item{stages}{An optional vector denoting the names of the variables within
the main vertical dataset coding for the stages of each individual in
occasions \emph{t}+1, \emph{t}, and \emph{t}-1. The names of stages in these
variables should match those used in the \code{stageframe} exactly. If left
blank, then \code{rlefko3()} will attempt to infer stages by matching values
of \code{alive}, \code{size}, \code{repst}, and \code{matst} to
characteristics noted in the associated \code{stageframe}.}

\item{alive}{A vector of names of binomial variables corresponding to status
as alive (\code{1}) or dead (\code{0}) in occasions \emph{t}+1 ans \emph{t},
respectively.}

\item{size}{A vector of names of variables coding the primary size variable
in occasions \emph{t}+1 and \emph{t}, respectively. Defaults to
\code{c("sizea3", "sizea2")}.}

\item{sizeb}{A vector of names of variables coding the secondary size
variable in occasions \emph{t}+1 and \emph{t}, respectively. Defaults to
\code{c(NA, NA)}.}

\item{sizec}{A vector of names of variables coding the tertiary size
variable in occasions \emph{t}+1 and \emph{t}, respectively. Defaults to
\code{c(NA, NA)}.}

\item{repst}{A vector of names of variables coding reproductive status in
occasions \emph{t}+1 and \emph{t}, respectively. Defaults to 
\code{c("repstatus3", "repstatus2")}. Must be supplied if \code{stages} is
not provided.}

\item{matst}{A vector of names of variables coding maturity status in
occasions \emph{t}+1 and \emph{t}, respectively. Defaults to
\code{c("matstatus3", "matstatus2")}. Must be supplied if \code{stages} is
not provided.}

\item{fec}{A vector of names of variables coding fecundity in occasions
\emph{t}+1 and \emph{t}, respectively. Defaults to \code{c("feca3", "feca2")}.}

\item{supplement}{An optional data frame of class \code{lefkoSD} that
provides supplemental data that should be incorporated into the MPM. Three
kinds of data may be integrated this way: transitions to be estimated via the
use of proxy transitions, transition overwrites from the literature or
supplemental studies, and transition multipliers for fecundity. This data
frame should be produced using the \code{\link{supplemental}()} function.
Should be used in place of or in addition to an overwrite table (see 
\code{overwrite} below) and a reproduction matrix (see \code{repmatrix}
below).}

\item{repmatrix}{An optional reproduction matrix. This matrix is composed
mostly of 0s, with non-zero entries acting as element identifiers and
multipliers for fecundity (with 1 equaling full fecundity). If left blank,
and no \code{supplement} is provided, then \code{rlefko2()} will assume that
all stages marked as reproductive produce offspring at 1x that of estimated
fecundity, and that offspring production will yield the first stage noted as
propagule or immature. To prevent this behavior, input just \code{0}, which
will result in fecundity being estimated only for transitions noted in
\code{supplement} above. Must be the dimensions of an ahistorical matrix.}

\item{overwrite}{An optional data frame developed with the
\code{\link{overwrite}()} function describing transitions to be overwritten
either with given values or with other estimated transitions. Note that this
function supplements overwrite data provided in \code{supplement}.}

\item{yearcol}{The variable name or column number corresponding to occasion 
\emph{t} in the dataset.}

\item{popcol}{The variable name or column number corresponding to the
identity of the population.}

\item{patchcol}{The variable name or column number corresponding to patch in
the dataset.}

\item{indivcol}{The variable name or column number coding individual
identity.}

\item{censorcol}{The variable name or column number denoting the censor
status. Only needed if \code{censor = TRUE}.}

\item{censorkeep}{The value of the censor variable denoting data elements to
keep. Defaults to \code{0}.}

\item{reduce}{A logical value denoting whether to remove historical stages
associated with only zero transitions. These are removed only if the
respective row and column sums in ALL matrices estimated equal 0. Defaults to
\code{FALSE}.}
}
\value{
If all inputs are properly formatted, then this function will return
an object of class \code{lefkoMat}, which is a list that holds the matrix
projection model and all of its metadata. Its structure is a list with the
following elements:

\item{A}{A list of full projection matrices in order of sorted populations,
patches, and occasions. All matrices output in the \code{matrix} class.}
\item{U}{A list of survival transition matrices sorted as in \code{A}. All 
matrices output in the \code{matrix} class.}
\item{F}{A list of fecundity matrices sorted as in \code{A}. All matrices 
output in the \code{matrix} class.}
\item{hstages}{A data frame matrix showing the pairing of ahistorical stages
used to create historical stage pairs. Set to NA for ahistorical matrices.}
\item{agestages}{A data frame showing age-stage pairs. In this function, it
is set to NA. Only used in output to function \code{aflefko2}().}
\item{ahstages}{A data frame detailing the characteristics of associated
ahistorical stages, in the form of a modified stageframe that includes
status as an entry stage through reproduction.}
\item{labels}{A data frame giving the population, patch, and year of each 
matrix in order.}
\item{matrixqc}{A short vector describing the number of non-zero elements
in \code{U} and \code{F} matrices, and the number of annual matrices.}
\item{dataqc}{A vector showing the numbers of individuals and rows in the
vertical dataset used as input.}
}
\description{
Function \code{rlefko2()} returns raw ahistorical MPMs, including the
associated component transition and fecundity matrices, a data frame
describing the ahistorical stages used, and a data frame describing the
population, patch, and occasion time associated with each matrix.
}
\section{Notes}{

The default behavior of this function is to estimate fecundity with regards
to transitions specified via associated fecundity multipliers in either
\code{supplement} or \code{repmatrix}. If both of these fields are left
empty, then fecundity will be estimated at full for all transitions leading
from reproductive stages to immature and propagule stages. However, if a
\code{supplement} is provided and a \code{repmatrix} is not, or if
\code{repmatrix} is set to 0, then only fecundity transitions noted in the
supplement will be set to non-zero values. To use the default behavior of
setting all reproductive stages to reproduce at full fecundity into immature
and propagule stages but also incorporate given or proxy survival
transitions, input those given and proxy transitions through the
\code{overwrite} options.

The reproduction matrix (field \code{repmatrix}) may only be supplied as
ahistorical. If provided as historical, then \code{rlefko2()} will fail and
produce an error.

Users may at times wish to estimate MPMs using a dataset incorporating
multiple patches or subpopulations. Should the aim of analysis be a general
MPM that does not distinguish these patches or subpopulations, the
\code{patchcol} variable should be left to \code{NA}, which is the default.
Otherwise the variable identifying patch needs to be named.

Input options including multiple variable names must be entered in the order
of variables in occasion \emph{t}+1 and \emph{t}. Rearranging the order WILL
lead to erroneous calculations, and may lead to fatal errors.

Although this function is capable of assigning stages given an input
stageframe, it lacks the power of \code{\link{verticalize3}()} and
\code{\link{historicalize3}()} in this regard. Users are strongly
encouraged to use the latter two functions for stage assignment.
}

\examples{
# Lathyrus example
data(lathyrus)

sizevector <- c(0, 100, 13, 127, 3730, 3800, 0)
stagevector <- c("Sd", "Sdl", "VSm", "Sm", "VLa", "Flo", "Dorm")
repvector <- c(0, 0, 0, 0, 0, 1, 0)
obsvector <- c(0, 1, 1, 1, 1, 1, 0)
matvector <- c(0, 0, 1, 1, 1, 1, 1)
immvector <- c(1, 1, 0, 0, 0, 0, 0)
propvector <- c(1, 0, 0, 0, 0, 0, 0)
indataset <- c(0, 1, 1, 1, 1, 1, 1)
binvec <- c(0, 100, 11, 103, 3500, 3800, 0.5)

lathframe <- sf_create(sizes = sizevector, stagenames = stagevector, 
  repstatus = repvector, obsstatus = obsvector, matstatus = matvector, 
  immstatus = immvector, indataset = indataset, binhalfwidth = binvec, 
  propstatus = propvector)

lathvert <- verticalize3(lathyrus, noyears = 4, firstyear = 1988, 
  patchidcol = "SUBPLOT", individcol = "GENET", blocksize = 9, 
  juvcol = "Seedling1988", sizeacol = "Volume88", repstracol = "FCODE88", 
  fecacol = "Intactseed88", deadacol = "Dead1988", nonobsacol = "Dormant1988", 
  stageassign = lathframe, stagesize = "sizea", censorcol = "Missing1988", 
  censorkeep = NA, censor = TRUE)

lathsupp2 <- supplemental(stage3 = c("Sd", "Sdl", "Sd", "Sdl"), 
  stage2 = c("Sd", "Sd", "rep", "rep"),
  givenrate = c(0.345, 0.054, NA, NA),
  multiplier = c(NA, NA, 0.345, 0.054),
  type = c(1, 1, 3, 3), stageframe = lathframe, historical = FALSE)

ehrlen2 <- rlefko2(data = lathvert, stageframe = lathframe, year = "all", 
  stages = c("stage3", "stage2"), supplement = lathsupp2, yearcol = "year2",
  indivcol = "individ")

summary(ehrlen2)

# Cypripedium example
rm(list=ls(all=TRUE))
data(cypdata)

sizevector <- c(0, 0, 0, 0, 0, 0, 1, 2.5, 4.5, 8, 17.5)
stagevector <- c("SD", "P1", "P2", "P3", "SL", "D", "XSm", "Sm", "Md", "Lg",
  "XLg")
repvector <- c(0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1)
obsvector <- c(0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1)
matvector <- c(0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1)
immvector <- c(0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0)
propvector <- c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0)
indataset <- c(0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1)
binvec <- c(0, 0, 0, 0, 0, 0.5, 0.5, 1, 1, 2.5, 7)

cypframe_raw <- sf_create(sizes = sizevector, stagenames = stagevector,
  repstatus = repvector, obsstatus = obsvector, matstatus = matvector,
  propstatus = propvector, immstatus = immvector, indataset = indataset,
  binhalfwidth = binvec)

cypraw_v1 <- verticalize3(data = cypdata, noyears = 6, firstyear = 2004,
  patchidcol = "patch", individcol = "plantid", blocksize = 4,
  sizeacol = "Inf2.04", sizebcol = "Inf.04", sizeccol = "Veg.04",
  repstracol = "Inf.04", repstrbcol = "Inf2.04", fecacol = "Pod.04",
  stageassign = cypframe_raw, stagesize = "sizeadded", NAas0 = TRUE,
  NRasRep = TRUE)

# Here we use supplemental() to provide overwrite and reproductive info
cypsupp2r <- supplemental(stage3 = c("SD", "P1", "P2", "P3", "SL", "D", 
    "XSm", "Sm", "SD", "P1"),
  stage2 = c("SD", "SD", "P1", "P2", "P3", "SL", "SL", "SL", "rep",
    "rep"),
  eststage3 = c(NA, NA, NA, NA, NA, "D", "XSm", "Sm", NA, NA),
  eststage2 = c(NA, NA, NA, NA, NA, "XSm", "XSm", "XSm", NA, NA),
  givenrate = c(0.10, 0.20, 0.20, 0.20, 0.25, NA, NA, NA, NA, NA),
  multiplier = c(NA, NA, NA, NA, NA, NA, NA, NA, 0.5, 0.5),
  type =c(1, 1, 1, 1, 1, 1, 1, 1, 3, 3),
  stageframe = cypframe_raw, historical = FALSE)

cypmatrix2r <- rlefko2(data = cypraw_v1, stageframe = cypframe_raw, 
  year = "all", patch = "all", stages = c("stage3", "stage2", "stage1"),
  size = c("size3added", "size2added"), supplement = cypsupp2r,
  yearcol = "year2", patchcol = "patchid", indivcol = "individ")

cypmatrix2r$A[[1]]

}
