% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.lmvar.R
\name{predict.lmvar}
\alias{predict.lmvar}
\title{Predictions for model matrices}
\usage{
\method{predict}{lmvar}(object, X_mu = NULL, X_sigma = NULL, mu = TRUE,
  sigma = TRUE, log = FALSE, interval = c("none", "confidence",
  "prediction"), level = 0.95, ...)
}
\arguments{
\item{object}{Object of class 'lmvar'}

\item{X_mu}{Model matrix for the expected values}

\item{X_sigma}{Model matrix for the logarithm of the standard deviations}

\item{mu}{Boolean, specifies whether or not to include the estimators and intervals for the expected values}

\item{sigma}{Boolean, specifies whether or not to include the estimators and intervals for
the standard deviations}

\item{log}{Boolean, specifies whether estimators and
intervals should be for \eqn{Y} (\code{log = FALSE}) or for \eqn{e^Y} (\code{log = TRUE}).}

\item{interval}{Character string, specifying the type of interval. Possible values are
\itemize{
\item "none" No interval, this is the default
\item "confidence" Confidence intervals for the estimators
\item "prediction" Prediction intervals
}}

\item{level}{Numeric value between 0 and 1, specifying the confidence level}

\item{...}{For compatibility with \code{\link[stats]{predict}} generic}
}
\value{
In the case \code{mu = FALSE} and \code{interval = "none"}: a numeric vector containing the estimators for
the standard deviation.

In the case \code{sigma = FALSE} and \code{interval = "none"}: a numeric vector containing the estimators for
the expected values.

In all other cases: a matrix with one column for each requested feature and one row for each observation. The column names are
\itemize{
\item \code{mu} Estimators for the expected value \eqn{\mu}
\item \code{sigma} Estimators for the standard deviation \eqn{\sigma}
\item \code{mu_lwr} Lower bound of the confidence interval for \eqn{\mu}
\item \code{mu_upr} Upper bound of the confidence interval for \eqn{\mu}
\item \code{sigma_lwr} Lower bound of the confidence interval for \eqn{\sigma}
\item \code{sigma_upr} Upper bound of the confidence interval for \eqn{\sigma}
\item \code{lwr} Lower bound of the prediction interval
\item \code{upr} Upper bound of the prediction interval
}
}
\description{
Estimators and confidence intervals  for the expected values
and standard deviations of the response vector \eqn{Y},
given model matrices \code{X_mu} and \code{X_sigma}. Prediction intervals for \eqn{Y}.
Alternatively, estimators and intervals can be for \eqn{e^Y}.

The estimators and intervals are based on the maximum likelihood-estimators
for \eqn{\beta_\mu} and \eqn{\beta_\sigma} and their covariance matrix
present in an 'lmvar' object.
}
\details{
When \code{X_mu = NULL}, the model matrix \eqn{X_\mu} is taken from \code{object}. Likewise, when
\code{X_sigma = NULL}, \eqn{X_\sigma} is taken from \code{object}.

Both \code{X_mu} and \code{X_sigma} must have column names. Column names are matched with the names of the elements of
\eqn{\beta_\mu} and \eqn{\beta_\sigma} in \code{object}. Columns with non-matching names are ignored. In case not all
names in \eqn{\beta_\mu} can be matched with a column in \code{X_mu}, a warning is given. The same is true for \eqn{\beta_\sigma}
and \code{X_sigma}.

\code{X_mu} can not have a column with the name  "(Intercept)". This column is added by \code{predict.lmvar} in case
it is present in \code{object}. Likewise,
\code{X_sigma} can not have a column with the name  "(Intercept_s)". It is added by \code{predict.lmvar} in case
it is present in \code{object}

Both matrices must be numeric and can not contain special values like
\code{NULL}, \code{NaN}, etc.

If \code{log = FALSE}, \code{predict.lmvar} returns
expected values and standard deviations for the observations \eqn{Y} corresponding to the model matrices \eqn{X_\mu}
and \eqn{X_\sigma}.

If \code{log = TRUE}, \code{predict.lmvar} returns expected values and standard deviations for \eqn{e^Y}.

The fit in \code{object} can be obtained under the constraint that the standard deviations \eqn{\sigma} are larger
than a minimum value (see the documentation of \code{\link{lmvar}}). However, there is no guarantee that the
values of \eqn{\sigma} given by \code{predict}, satisfy the same constraint. This depends entirely on
\code{X_sigma}.

Confidence intervals are calculated under the asumption of asymptotic normality. This asumption holds when the number
of observations is large.
Intervals must be treated cautiously in case of a small number of observations. Intervals can also be unreliable if
\code{object} was created with a constraint on the minimum values of the standard deviations \eqn{\sigma}.

\code{predict.lmvar} with \code{X_mu = NULL} and \code{X_sigma = NULL} is equivalent to the function
\code{\link{fitted.lmvar}}.
}
\examples{
# As example we use the dataset 'attenu' from the library 'datasets'. The dataset contains
# the response variable 'accel' and two explanatory variables 'mag'  and 'dist'.
library(datasets)

# Create the model matrix for the expected values
X = cbind(attenu$mag, attenu$dist)
colnames(X) = c("mag", "dist")

# Create the model matrix for the standard deviations.
X_s = cbind(attenu$mag, 1 / attenu$dist)
colnames(X_s) = c("mag", "dist_inv")

# Create the response vector
y = attenu$accel

# Carry out the fit
fit = lmvar(y, X, X_s)

# Calculate the expected values and standard deviations of 'accel'
# for the current magnitudes and distances
predict(fit)

# Calculate the expected values and standard deviations of 'accel' for earthquakes
# with a 10\% larger magnitude, at the current distances
XP = cbind(1.1 * attenu$mag, attenu$dist)
colnames(XP) = c("mag", "dist")

XP_s = cbind(1.1 * attenu$mag, 1 / attenu$dist)
colnames(XP_s) = c("mag", "dist_inv")

predict(fit, XP, XP_s)

# Calculate only the expected values
predict(fit, XP, XP_s, sigma = FALSE)

# Calculate only the standard deviations
predict(fit, XP, XP_s, mu = FALSE)

# Calculate the expected values and their 95\% confidence intervals
predict(fit, XP, XP_s, sigma = FALSE, interval = "confidence")

# Calculate the standard deviations and their 90\% confidence intervals
predict(fit, XP, XP_s, mu = FALSE, interval = "confidence", level = 0.9)

# Calculate the expected values and the 90\% prediction intervals of 'accel'
predict(fit, XP, XP_s, sigma = FALSE, interval = "prediction", level = 0.9)

# Change the model and fit the log of 'accel'
y = log(attenu$accel)
fit_log = lmvar(y, X, X_s)

# Calculate the expected values and standard deviations of the log of 'accel'
predict(fit_log, XP, XP_s)

# Calculate the expected values and standard deviations of 'accel'
predict(fit_log, XP, XP_s, log = TRUE)

# Calculate the expected values and standard deviations of 'accel',
# as well as their 99\% confidence intervals
predict(fit_log, XP, XP_s, log = TRUE, interval = "confidence", level = 0.99)

}
\seealso{
\code{\link{coef.lmvar}} and \code{\link[stats]{confint}} for maximum likelihood estimators
and confidence intervals for \eqn{\beta_\mu} and \eqn{\beta_\sigma}.

\code{\link{fitted.lmvar}} is equivalent to \code{predict.lmvar} with \code{X_mu = NULL} and
\code{X_sigma = NULL}.
}
