
#' Plot gene tracks
#' 
#' Plot gene annotation tracks from `ensembldb` data using ggplot2 and grid.
#' 
#' @param loc Object of class 'locus' generated by [locus()].
#' @param filter_gene_name Vector of gene names to display.
#' @param filter_gene_biotype Vector of gene biotypes to be filtered. Use
#' [ensembldb::listGenebiotypes()] to display possible biotypes. For example, 
#' `ensembldb::listGenebiotypes(EnsDb.Hsapiens.v75)`
#' @param border Logical whether a bounding box is plotted.
#' @param cex.axis Specifies font size for axis numbering.
#' @param cex.lab Specifies font size for axis titles.
#' @param cex.text Font size for gene text.
#' @param gene_col Colour for gene lines.
#' @param exon_col Fill colour for exons.
#' @param exon_border Border line colour outlining exons (or genes if
#'   `showExons` is `FALSE`). Set to `NA` for no border.
#' @param showExons Logical whether to show exons or simply show whole gene as a
#'   rectangle. If `showExons = FALSE` colours are specified by `exon_border`
#'   for rectangle border and `gene_col` for the fill colour.
#' @param maxrows Specifies maximum number of rows to display in gene annotation
#'   panel.
#' @param text_pos Character value of either 'top' or 'left' specifying
#'   placement of gene name labels.
#' @param xticks Logical whether x axis ticks and numbers are plotted.
#' @param xlab Title for x axis. Defaults to chromosome `seqname` specified in
#'   `locus`.
#' @param highlight Vector of genes to highlight.
#' @param highlight_col Single colour or vector of colours for highlighted
#'   genes.
#' @param blanks Controls handling of genes with blank names: `"fill"` replaces
#'   blank gene symbols with ensembl gene ids. `"hide"` hides genes which are
#'   missing gene symbols.
#' @details
#' This function is called by [locus_ggplot()], and in turn it calls
#' [genetracks_grob()]. It can be used to plot the gene annotation tracks on
#' their own as a ggplot2 object.
#' 
#' `gene_col`, `exon_col` and `exon_border` set colours for all genes, while
#' `highlight` and `highlight_col` can optionally be used together to highlight
#' specific genes of interest. For full control over every single gene, users
#' can add columns `gene_col`, `exon_col` and `exon_border` to the `TX` object
#' within the 'locus' object. Columns added to `TX` override their equivalent 
#' arguments.
#' @seealso [locus_ggplot()] [genetracks_grob()]
#' @return A ggplot2 object.
#' @examples
#' if(require(EnsDb.Hsapiens.v75)) {
#' data(SLE_gwas_sub)
#' loc <- locus(SLE_gwas_sub, gene = 'IRF5', flank = c(7e4, 2e5), LD = "r2",
#'              ens_db = "EnsDb.Hsapiens.v75")
#' gg_genetracks(loc)
#' }
#' @importFrom ggplot2 xlab
#' @importFrom gggrid grid_panel
#' @export

gg_genetracks <- function(loc,
                          filter_gene_name = NULL,
                          filter_gene_biotype = NULL,
                          border = FALSE,
                          cex.axis = 1,
                          cex.lab = 1,
                          cex.text = 0.7,
                          gene_col = ifelse(showExons, 'blue4', 'skyblue'),
                          exon_col = 'blue4',
                          exon_border = 'blue4',
                          showExons = TRUE,
                          maxrows = NULL,
                          text_pos = 'top',
                          xticks = TRUE,
                          xlab = NULL,
                          highlight = NULL,
                          highlight_col = "red",
                          blanks = c("fill", "hide")) {
  if (!inherits(loc, "locus")) stop("Object of class 'locus' required")
  blanks <- match.arg(blanks)
  g <- genetracks_grob(loc,
                       filter_gene_name,
                       filter_gene_biotype,
                       border,
                       cex.text,
                       gene_col,
                       exon_col,
                       exon_border,
                       showExons,
                       maxrows,
                       text_pos,
                       highlight, highlight_col,
                       blanks)
  if (is.null(xlab) & xticks) xlab <- paste("Chromosome", loc$seqname, "(Mb)")
  
  g2 <- ggplot(data.frame(x = NA),
               aes(xmin = loc$xrange[1] / 1e6, xmax = loc$xrange[2] / 1e6)) +
    (if (!is.null(g)) gggrid::grid_panel(g)) +
    xlab(xlab) +
    theme_classic() +
    theme(axis.text = element_text(colour = "black", size = 10 * cex.axis),
          axis.title = element_text(size = 10 * cex.lab),
          axis.line.y = element_blank())
  if (!xticks) {
    g2 <- g2 +
      theme(axis.line.x = element_blank(),
            axis.ticks.x = element_blank(),
            axis.text.x = element_blank())
  }
  g2
}

