\name{logcon-package}
\alias{logcon-package}
\alias{logcon}
\alias{log-concave}
\alias{logcondens}
\docType{package}
\title{Estimate a Log-Concave Probability Density from iid Observations}
\description{The main function of this package is \code{\link{logConDens}}: A function to compute the maximum likelihood estimator (MLE) of a log-concave density from one-dimensional i.i.d. observations as well as the kernel smoothed version derived from it. A list of additional functions that can be used to compute quantities relevant in that context can be found below.

Two algorithms are offered to compute the estimate: An active set (\code{\link{logConDens}}) and an iterative algorithm based on the pool-adjacent-violaters algorithm (\code{\link{icmaLogCon}}). The latter of these functions is only part of this package for historical reasons: it was the first method that was proposed to estimate a log-concave density, see Rufibach (2007). The more efficient way of computing the estimate is via an active set algorithm. The smoothed versions of the log-concave density and distribution function estimates discussed in Section 3 of Duembgen and Rufibach (2009) are available in the function \code{\link{evaluateLogConDens}}.

To compute a log-concave density, CDF, and survival function from interval- or right-censored observations use the package \pkg{logconcens}. A log-concave probability mass function can be computed using the package \pkg{logcondiscr}, see Balabdaoui et al (2012) for details. For the computation of a log-concave estimate in any dimension the package \pkg{LogConDEAD} can be used.}

\details{
\tabular{ll}{
Package: \tab logcondens\cr
Type: \tab Package\cr
Version: \tab 2.0.8 \cr
Date: \tab 2012-10-09 \cr
License: \tab GPL (>=2) \cr
}

The following additional functions and datasets are provided in the package:

\code{\link{evaluateLogConDens}} Computes the value of the estimated log-density, density, and distribution function 
of the MLE and the smoothed estimator at a given \code{x0}. 

\code{\link{quantilesLogConDens}} Computes quantiles of the estimated distribution function at a given \code{x0}. 

\code{\link{intECDF}} Compute the integrated empirical distribution function of the observations at a given \code{x0}. 

\code{\link{intF}} Compute the integral of the distribution function corresponding to the log-concave density 
estimator at a given \code{x0}. 

\code{\link{logconTwoSample}} Compute a permutation test for the difference between two distribution functions. 

\code{\link{logConROC}} Compute ROC curve based on log-concave density estimates within cases and controls.

\code{\link{confIntBootLogConROC_t0}} Compute bootstrap confidence intervals at given false positive fractions (= 1 - specificity) for the ROC curve based on log-concave density estimates.

\code{\link{logConCI}} Compute a confidence interval for the value of the true density at a fixed point, based on the theory developed in Balabdaoui et al (2009). This function was contributed by Mahdis Azadbakhsh and Hanna Jankowski, see Azadbakhsh et al (2012).

\code{\link{isoMean}} Compute the weighted least squares regression under a monotonicity constraint (the Grenander estimator).
This function is used as part of \code{\link{icmaLogCon}} but is also of independent interest. 

The following datasets have been used in several publications to illustrate log-concave density estimation and are
therefore included in this package:

\code{\link{reliability}} Dataset that contains the data analyzed in Duembgen and Rufibach (2009, Figure 2).
See the help file for \code{\link{logConDens}} for the analysis of this data. 

\code{\link{brightstar}} Dataset that contains the data analyzed in Mizera and Koenker (2009, Section 5). The 
sample consists of measurements of radial and rotational velocities for the stars from the Bright Star Catalog, see 
Hoffleit and Warren (1991). 

\code{\link{pancreas}} Data from pancreatic cancer serum biomarker study, first published by Wieand et al (1989). 
Contains data on serum measurements for a cancer antigen (CA-125) and
a carbohydrate antigen (CA19.9) both of which are measured on a continuous non-negative scale. The measurements were taken within a case-control
study on 90 cases with pancreatic cancer and 51 controls who did not have cancer but pancreatitis. 

A \code{\link{print}} and \code{\link{summary}} for objects of class \code{dlc}, generated by  \code{\link{logConDens}}, are available as well.
}
\author{
Kaspar Rufibach (maintainer), \email{kaspar.rufibach@gmail.com}, \cr \url{http://www.kasparrufibach.ch}

Lutz Duembgen, \email{duembgen@stat.unibe.ch}, \cr \url{http://www.staff.unibe.ch/duembgen}

Kaspar Rufibach gratefully acknowledges support by the Swiss National Science Foundation SNF, \url{http://www.snf.ch}.

MatLab code with an implementation of the active set algorithm is available on request from Lutz Duembgen.}
\references{
Azadbakhsh, M., Jankowski, H. and Gao, X. (2012).
Computing confidence intervals for log-concave densities. 
Preprint.

Balabdaoui, F., Jankowski, H., and Rufibach, K., Pavlides, M. (2012).
Asymptotics of the discrete log-concave maximum likelihood estimator and related applications.
Preprint.

Baladbaoui, F., Rufibach, K. and Wellner, J. (2009).
Limit distribution theory for maximum likelihood estimation of a log-concave density.
\emph{Ann. Statist.}, \bold{37(3)}, 1299--1331.

Duembgen, L., Huesler, A. and Rufibach, K. (2010).
Active set and EM algorithms for log-concave densities based on complete and censored data.
Technical report 61, IMSV, Univ. of Bern, available at \url{http://arxiv.org/abs/0707.4643}.

Duembgen, L. and Rufibach, K. (2009).
Maximum likelihood estimation of a log--concave density and its distribution function: basic properties and uniform consistency. 
\emph{Bernoulli}, \bold{15(1)}, 40--68.  

Duembgen, L. and Rufibach, K. (2011).
logcondens: Computations Related to Univariate Log-Concave Density Estimation. 
\emph{Journal of Statistical Software}, \bold{39(6)}, 1--28. \url{http://www.jstatsoft.org/v39/i06}

Hoffleit, D., Warren, W.H. (1991).
\emph{The Bright Star Catalog.}
Yale University Observatory, New Heaven.  

Mizera, I., Koenker, R. (2010).
Quasi-concave density estimation.
\emph{Ann. Statist.}, \bold{38(5)}, 2998--3027. 

Rufibach K. (2006). 
\emph{Log-concave Density Estimation and Bump Hunting for i.i.d. Observations.}
PhD Thesis, University of Bern, Switzerland and Georg-August University of Goettingen, Germany, 2006.
\cr Available at \url{http://www.stub.unibe.ch/download/eldiss/06rufibach_k.pdf}.

Rufibach, K. (2007).
Computing maximum likelihood estimators of a log-concave density function.
\emph{J. Stat. Comput. Simul.} \bold{77}, 561--574.

Rufibach, K. (2012).
A smooth ROC curve estimator based on log-concave density estimates.
\emph{Int. J. Biostat.}, \bold{8}(1), 1--29.
}

\examples{
## estimate gamma density
set.seed(1977)
x <- rgamma(100, 2, 1)
res <- logConDens(x, smoothed = FALSE, print = TRUE)
summary(res)

## compare performance to ICMA
res2 <- icmaLogCon(x, T1 = 2000, robustif = TRUE, print = TRUE)

res$L
res2$L

## plot resulting functions
par(mfrow = c(2, 2), mar = c(3, 2, 1, 2))
plot(res, which = "density")
plot(res, which = "log-density")
plot(res, which = "CDF")
plot(res$x, res$H, type = 'l'); rug(res$x)

## compute function values at an arbitrary point
x0 <- (res$x[50] + res$x[51]) / 2
evaluateLogConDens(x0, res)

## compute 0.5 quantile of Fhat
quantilesLogConDens(0.5, res)
}
\keyword{htest}
\keyword{nonparametric}
