% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mts.R
\name{mts_create_tileset}
\alias{mts_create_tileset}
\title{Create a tileset with the Mapbox Tiling Service API}
\usage{
mts_create_tileset(
  tileset_name,
  username,
  recipe,
  request_name = tileset_name,
  access_token = NULL
)
}
\arguments{
\item{tileset_name}{The name of the MTS tileset you intend to create}

\item{username}{Your Mapbox username}

\item{recipe}{An MTS recipe, created with \code{mts_make_recipe()}}

\item{request_name}{The name of the request; defaults to the tileset name}

\item{access_token}{Your Mapbox access token}
}
\value{
The response from the API, formatted as an R list.
}
\description{
After you've uploaded your spatial data to your Mapbox account with \code{mts_create_source} and prepared a valid recipe with \code{mts_make_recipe()}, you can use your source and recipe to create a vector tileset.  This tileset will be hosted at your Mapbox account.  Once created successfully, you will need to publish the tileset using \code{mts_publish_tileset} to use it in Mapbox Studio, Mapbox GL JS, or an R package that can read Mapbox tilesets.
}
\examples{
\dontrun{
library(tidycensus)
library(mapboxapi)
options(tigris_use_cache = TRUE)

# Get the national data on median age
us_median_age_tract <- get_acs(
  geography = "tract",
  variables = "B01002_001",
  state = c(state.abb, "DC"),
  year = 2020,
  geometry = TRUE
)

# Get it for counties as well
us_median_age_county <- get_acs(
  geography = "county",
  variables = "B01002_001",
  year = 2020,
  geometry = TRUE
)

# Create a source from the datasets
mts_create_source(data = us_median_age_tract,
                  tileset_id = "us_median_age_tract",
                  username = "your_mapbox_username")

mts_create_source(data = us_median_age_county,
                  tileset_id = "us_median_age_county",
                  username = "your_mapbox_username")

# Build out the recipe.  First, create a recipe layer with
# appropriate options.  We'll want a larger tile size and to restrict the minzoom
# to 4; a maxzoom of 12 will be fine as we can overzoom beyond that
#
# Your source ID will be returned by `mts_create_source()`, so use that value
tract_layer <- recipe_layer(
  source = "mapbox://tileset-source/your_mapbox_username/us_median_age_tract",
  minzoom = 4,
  maxzoom = 12,
  tiles = tile_options(layer_size = 2500)
)

county_layer <- recipe_layer(
  source = "mapbox://tileset-source/your_mapbox_username/us_median_age_county",
  minzoom = 2,
  maxzoom = 5
)

recipe <- mts_make_recipe(tracts = tract_layer, counties = county_layer)

# Validate the recipe
mts_validate_recipe(recipe)

# Create a tileset from the recipe
mts_create_tileset(tileset_name = "median_age_acs",
                   username = "your_mapbox_username",
                   recipe = recipe)

# Publish the tileset
mts_publish_tileset(tileset_name = "median_age_acs",
                    username = "your_mapbox_username")

# If necessary, update the recipe
mts_update_recipe(tileset_name = "median_age_acs",
                  username = "your_mapbox_username",
                  recipe = new_recipe)

# Publish the tileset again after you've updated the recipe
mts_publish_tileset(tileset_name = "median_age_acs",
                    username = "your_mapbox_username")
}
}
\seealso{
\url{https://docs.mapbox.com/api/maps/mapbox-tiling-service/#create-a-tileset}
}
