# Copyright (C) 2023-2025 Hibiki AI Limited <info@hibiki-ai.com>
#
# This file is part of mirai.
#
# mirai is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# mirai is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# mirai. If not, see <https://www.gnu.org/licenses/>.

# mirai ------------------------------------------------------------------------

#' Launch Daemon
#'
#' `launch_local` spawns a new background `Rscript` process calling [daemon()]
#' with the specified arguments.
#'
#' These functions may be used to re-launch daemons that have exited after
#' reaching time or task limits.
#'
#' Daemons must already be set for launchers to work.
#'
#' The generated command contains the argument `rs` specifying the length 7
#' L'Ecuyer-CMRG random seed supplied to the daemon. The values will be
#' different each time the function is called.
#'
#' @inheritParams mirai
#' @param n integer number of daemons.
#'
#'   **or** for `launch_remote` only, a 'miraiCluster' or 'miraiNode'.
#' @param ... (optional) arguments passed through to [daemon()]. These include
#'   `autoexit`, `cleanup`, `output`, `maxtasks`, `idletime` and `walltime`.
#'   Only supply to override arguments originally provided to [daemons()],
#'   otherwise those will be used instead.
#' @param tls \[default NULL\] required for secure TLS connections over
#'   'tls+tcp://'. Zero-configuration TLS certificates generated by [daemons()]
#'   are automatically passed to the daemon, without requiring to be specified
#'   here. Otherwise, supply **either** the character path to a file containing
#'   X.509 certificate(s) in PEM format, comprising the certificate authority
#'   certificate chain, **or** a length 2 character vector comprising \[i\] the
#'   certificate authority certificate chain and \[ii\] the empty string `""`.
#'
#' @return For **launch_local**: Integer number of daemons launched.
#'
#' @examplesIf interactive()
#' daemons(url = host_url(), dispatcher = FALSE)
#' status()
#' launch_local(1L, cleanup = FALSE)
#' launch_remote(1L, cleanup = FALSE)
#' Sys.sleep(1)
#' status()
#' daemons(0)
#'
#' daemons(url = host_url(tls = TRUE))
#' status()
#' launch_local(2L, output = TRUE)
#' Sys.sleep(1)
#' status()
#' daemons(0)
#'
#' @export
#'
launch_local <- function(n = 1L, ..., tls = NULL, .compute = "default") {

  if (missing(.compute)) .compute <- .[["cp"]]
  envir <- ..[[.compute]]
  is.null(envir) && stop(._[["daemons_unset"]])
  url <- envir[["urls"]]
  write_args <- if (length(envir[["msgid"]])) wa3 else wa2
  dots <- if (missing(..1)) envir[["dots"]] else parse_dots(...)
  output <- attr(dots, "output")
  if (is.null(tls)) tls <- envir[["tls"]]
  for (i in seq_len(n))
    launch_daemon(write_args(url, dots, next_stream(envir), tls), output)
  n

}

#' Launch Daemon
#'
#' `launch_remote` returns the shell command for deploying daemons as a
#' character vector. If a configuration generated by [remote_config()] or
#' [ssh_config()] is supplied then this is used to launch the daemon on the
#' remote machine.
#'
#' @param remote required only for launching remote daemons, a configuration
#'   generated by [remote_config()] or [ssh_config()]. An empty
#'   [remote_config()] does not effect any daemon launches but returns the shell
#'   commands for deploying manually on remote machines.
#'
#' @return For **launch_remote**: A character vector of daemon launch commands,
#'   classed as 'miraiLaunchCmd'. The printed output may be copy / pasted
#'   directly to the remote machine.
#'
#' @rdname launch_local
#' @export
#'
launch_remote <- function(n = 1L, remote = remote_config(), ..., tls = NULL, .compute = "default") {

  if (missing(.compute)) .compute <- .[["cp"]]
  if (!is.numeric(n) && inherits(n, c("miraiCluster", "miraiNode"))) {
    .compute <- attr(n, "id")
    n <- max(length(n), 1L)
  }
  n <- as.integer(n)
  envir <- ..[[.compute]]
  is.null(envir) && stop(._[["daemons_unset"]])
  url <- envir[["urls"]]
  write_args <- if (length(envir[["msgid"]])) wa3 else wa2
  dots <- if (missing(..1)) envir[["dots"]] else parse_dots(...)
  if (is.null(tls)) tls <- envir[["tls"]]

  command <- remote[["command"]]
  rscript <- remote[["rscript"]]
  quote <- remote[["quote"]]

  if (length(command)) {

    args <- remote[["args"]]

    if (is.list(args)) {

      tunnel <- remote[["tunnel"]]

      if (tunnel) {
        purl <- parse_url(url)
        purl[["hostname"]] == "127.0.0.1" || stop(._[["localhost"]])
        prefix <- sprintf("-R %s:127.0.0.1:%s", purl[["port"]], purl[["port"]])
        for (i in seq_along(args))
          args[[i]][1L] <- sprintf("%s %s", prefix, args[[i]][1L])
      }

      if (length(args) == 1L) {
        args <- args[[1L]]

      } else if (n == 1L || n == length(args)) {

        cmds <- character(length(args))
        for (i in seq_along(args))
          cmds[i] <- sprintf("%s -e %s", rscript, write_args(url, dots, next_stream(envir), tls))

        for (i in seq_along(args))
          system2(command, args = `[<-`(args[[i]], find_dot(args[[i]]), if (quote) shQuote(cmds[i]) else cmds[i]), wait = FALSE)

        return(`class<-`(cmds, "miraiLaunchCmd"))

      } else {
        stop(._[["arglen"]])
      }

    }
  }

  cmds <- character(n)
  for (i in seq_len(n))
    cmds[i] <- sprintf("%s -e %s", rscript, write_args(url, dots, next_stream(envir), tls))

  if (length(command))
    for (cmd in cmds)
      system2(command, args = `[<-`(args, find_dot(args), if (quote) shQuote(cmd) else cmd), wait = FALSE)

  `class<-`(cmds, "miraiLaunchCmd")

}

#' Generic and SSH Remote Launch Configuration
#'
#' `remote_config` provides a flexible generic framework for generating the
#' shell commands to deploy daemons remotely.
#'
#' @param command the command used to effect the daemon launch on the remote
#'   machine as a character string (e.g. `"ssh"`). Defaults to `"ssh"` for
#'   `ssh_config`, although may be substituted for the full path to a specific
#'   SSH application. The default NULL for `remote_config` does not carry out
#'   any launches, but causes [launch_remote()] to return the shell commands for
#'   manual deployment on remote machines.
#' @param args (optional) arguments passed to `command`, as a character vector
#'   that must include `"."` as an element, which will be substituted for the
#'   daemon launch command. Alternatively, a list of such character vectors to
#'   effect multiple launches (one for each list element).
#' @param rscript (optional) name / path of the Rscript executable on the remote
#'   machine. The default assumes `"Rscript"` is on the executable search
#'   path. Prepend the full path if necessary. If launching on Windows,
#'   `"Rscript"` should be replaced with `"Rscript.exe"`.
#' @param quote \[default FALSE\] logical value whether or not to quote the
#'   daemon launch command (not required for Slurm `"srun"` for example, but
#'   required for Slurm `"sbatch"` or `"ssh"`).
#'
#' @return A list in the required format to be supplied to the `remote` argument
#'   of [launch_remote()], [daemons()], or [make_cluster()].
#'
#' @examples
#' # Slurm srun example
#' remote_config(
#'   command = "srun",
#'   args = c("--mem 512", "-n 1", "."),
#'   rscript = file.path(R.home("bin"), "Rscript")
#' )
#'
#' # Slurm sbatch requires 'quote = TRUE'
#' remote_config(
#'   command = "sbatch",
#'   args = c("--mem 512", "-n 1", "--wrap", "."),
#'   rscript = file.path(R.home("bin"), "Rscript"),
#'   quote = TRUE
#' )
#'
#' # SSH also requires 'quote = TRUE'
#' remote_config(
#'   command = "/usr/bin/ssh",
#'   args = c("-fTp 22 10.75.32.90", "."),
#'   quote = TRUE
#' )
#'
#' # can be used to start local dameons with special configurations
#' remote_config(
#'   command = "Rscript",
#'   rscript = "--default-packages=NULL --vanilla"
#' )
#'
#' @export
#'
remote_config <- function(command = NULL, args = c("", "."), rscript = "Rscript", quote = FALSE) {

  if (is.list(args)) lapply(args, find_dot) else find_dot(args)
  list(command = command, args = args, rscript = rscript, quote = quote, tunnel = FALSE)

}

#' SSH Remote Launch Configuration
#'
#' `ssh_config` generates a remote configuration for launching daemons over
#' SSH, with the option of SSH tunnelling.
#'
#' @param remotes the character URL or vector of URLs to SSH into, using the
#'   'ssh://' scheme and including the port open for SSH connections (defaults
#'   to 22 if not specified), e.g. 'ssh://10.75.32.90:22' or 'ssh://nodename'.
#' @param tunnel \[default FALSE\] logical value, whether to use SSH tunnelling.
#'   If TRUE, requires the [daemons()] `url` hostname to be '127.0.0.1'. See
#'   the 'SSH Tunnelling' section below for further details.
#' @param timeout \[default 10\] maximum time allowed for connection setup in
#'   seconds.
#'
#' @section SSH Direct Connections:
#'
#' The simplest use of SSH is to execute the daemon launch command on a remote
#' machine, for it to dial back to the host / dispatcher URL.
#'
#' It is assumed that SSH key-based authentication is already in place. The
#' relevant port on the host must also be open to inbound connections from the
#' remote machine, and is hence suitable for use within trusted networks.
#'
#' @section SSH Tunnelling:
#'
#' Use of SSH tunnelling provides a convenient way to launch remote daemons
#' without requiring the remote machine to be able to access the host. Often
#' firewall configurations or security policies may prevent opening a port to
#' accept outside connections.
#'
#' In these cases SSH tunnelling offers a solution by creating a tunnel once the
#' initial SSH connection is made. For simplicity, this SSH tunnelling
#' implementation uses the same port on both host and daemon. SSH key-based
#' authentication must already be in place, but no other configuration is
#' required.
#'
#' To use tunnelling, set the hostname of the [daemons()] `url` argument to be
#' '127.0.0.1'. Using [local_url()] with `tcp = TRUE` also does this for you.
#' Specifying a specific port to use is optional, with a random ephemeral port
#' assigned otherwise. For example, specifying 'tcp://127.0.0.1:5555' uses the
#' local port '5555' to create the tunnel on each machine. The host listens
#' to '127.0.0.1:5555' on its machine and the remotes each dial into
#' '127.0.0.1:5555' on their own respective machines.
#'
#' This provides a means of launching daemons on any machine you are able to
#' access via SSH, be it on the local network or the cloud.
#'
#' @examples
#' # direct SSH example
#' ssh_config(c("ssh://10.75.32.90:222", "ssh://nodename"), timeout = 5)
#'
#' # SSH tunnelling example
#' ssh_config(c("ssh://10.75.32.90:222", "ssh://nodename"), tunnel = TRUE)
#'
#' \dontrun{
#'
#' # launch 2 daemons on the remote machines 10.75.32.90 and 10.75.32.91 using
#' # SSH, connecting back directly to the host URL over a TLS connection:
#' daemons(
#'   url = host_url(tls = TRUE),
#'   remote = ssh_config(c("ssh://10.75.32.90:222", "ssh://10.75.32.91:222"))
#' )
#'
#' # launch 2 daemons on the remote machine 10.75.32.90 using SSH tunnelling:
#' daemons(
#'   n = 2,
#'   url = local_url(tcp = TRUE),
#'   remote = ssh_config("ssh://10.75.32.90", tunnel = TRUE)
#' )
#' }
#'
#' @rdname remote_config
#' @export
#'
ssh_config <- function(remotes, tunnel = FALSE, timeout = 10, command = "ssh", rscript = "Rscript") {

  premotes <- lapply(remotes, parse_url)
  hostnames <- lapply(premotes, .subset2, "hostname")
  ports <- lapply(premotes, .subset2, "port")

  args <- vector(mode = "list", length = length(remotes))
  for (i in seq_along(args)) {
    args[[i]] <- c(
      sprintf("-o ConnectTimeout=%s -fTp %s", as.character(timeout), ports[[i]]),
      hostnames[[i]],
      "."
    )
  }

  list(command = command, args = args, rscript = rscript, quote = TRUE, tunnel = isTRUE(tunnel))

}

#' URL Constructors
#'
#' `host_url` constructs a valid host URL (at which daemons may connect)
#' based on the computer's hostname. This may be supplied directly to the `url`
#' argument of [daemons()].
#'
#' `host_url` relies on using the host name of the computer rather than an IP
#' address and typically works on local networks, although this is not always
#' guaranteed. If unsuccessful, substitute an IPv4 or IPv6 address in place of
#' the hostname.
#'
#' `local_url` generates a random URL for the platform's default inter-process
#' communications transport: abstract Unix domain sockets on Linux, Unix domain
#' sockets on MacOS, Solaris and other POSIX platforms, and named pipes on
#' Windows.
#'
#' @param tls \[default FALSE\] logical value whether to use TLS in which case
#'   the scheme used will be 'tls+tcp://'.
#' @param port \[default 0\] numeric port to use. `0` is a wildcard value that
#'   automatically assigns a free ephemeral port. For `host_url`, this port
#'   should be open to connections from the network addresses the daemons are
#'   connecting from. For `local_url`, is only taken into account if
#'   `tcp = TRUE`.
#'
#' @return A character string comprising a valid URL.
#'
#' @examples
#' host_url()
#' host_url(tls = TRUE)
#' host_url(tls = TRUE, port = 5555)
#'
#' @export
#'
host_url <- function(tls = FALSE, port = 0)
  sprintf(
    "%s://%s:%s",
    if (tls) "tls+tcp" else "tcp", Sys.info()[["nodename"]], as.character(port)
  )

#' URL Constructors
#'
#' `local_url` constructs a URL suitable for local daemons, or for use with
#' SSH tunnelling. This may be supplied directly to the `url` argument of
#' [daemons()].
#'
#' @param tcp \[default FALSE\] logical value whether to use a TCP connection.
#'   This must be used for SSH tunnelling.
#'
#' @examples
#' local_url()
#' local_url(tcp = TRUE)
#' local_url(tcp = TRUE, port = 5555)
#'
#' @rdname host_url
#' @export
#'
local_url <- function(tcp = FALSE, port = 0)
  if (tcp)
    sprintf("tcp://127.0.0.1:%d", as.integer(port)) else
      sprintf("%s%s", .urlscheme, random(12L))

#' @export
#'
print.miraiLaunchCmd <- function(x, ...) {

  for (i in seq_along(x))
    cat(sprintf("[%d]\n%s\n\n", i, x[i]), file = stdout())
  invisible(x)

}

# internals --------------------------------------------------------------------

find_dot <- function(args) {
  sel <- args == "."
  any(sel) || stop(._[["dot_required"]], call. = FALSE)
  sel
}
