% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/observeNetwork.R
\name{observeNetwork}
\alias{observeNetwork}
\title{Observe a network partially according to a given sampling design}
\usage{
observeNetwork(
  adjacencyMatrix,
  sampling,
  parameters,
  clusters = NULL,
  covariates = NULL,
  similarity = missSBM:::l1_similarity,
  intercept = 0
)
}
\arguments{
\item{adjacencyMatrix}{The N x N adjacency matrix of the network to sample.}

\item{sampling}{The sampling design used to observe the adjacency matrix, see details.}

\item{parameters}{The sampling parameters (adapted to each sampling, see details).}

\item{clusters}{An optional clustering membership vector of the nodes. Only necessary for block samplings.}

\item{covariates}{An optional list with M entries (the M covariates). If the covariates are node-centered,
each entry of \code{covariates}. must be a size-N vector;  if the covariates are dyad-centered, each entry
of \code{covariates} must be N x N matrix.}

\item{similarity}{An optional function to compute similarities between node covariates. Default is
\code{missSBM:::l1_similarity}, that is, -abs(x-y). Only relevant when the covariates are node-centered.}

\item{intercept}{An optional intercept term to be added in case of the presence of covariates. Default is 0.}
}
\value{
an adjacency matrix with the same dimension as the input, yet with additional NAs.
}
\description{
This function draws observations in an adjacency matrix according to a given network sampling design.
}
\details{
The list of parameters \code{control} tunes more advanced features, such as the
initialization, how covariates are handled in the model, and the variational EM algorithm:
\itemize{
\item{"useCovSBM": }{logical. If \code{covariates} is not null, should they be used for the
for the SBM inference (or just for the sampling)? Default is TRUE.}
\item{"clusterInit": }{Initial method for clustering: either a character in "hierarchical", "spectral"
or "kmeans", or a list with \code{length(vBlocks)} vectors, each with size
\code{ncol(adjacencyMatrix)},  providing a user-defined clustering. Default is "spectral".}
\item{"similarity": }{An R x R -> R function to compute similarities between node covariates. Default is
\code{missSBM:::l1_similarity}, that is, -abs(x-y). Only relevant when the covariates are node-centered
(i.e. \code{covariates} is a list of size-N vectors).}
\item{"threshold": }{V-EM algorithm stops stop when an optimization step changes the objective function
by less than threshold. Default is 1e-3.}
\item{"maxIter": }{V-EM algorithm stops when the number of iteration exceeds maxIter.
Default is 100 with no covariate, 50 otherwise.}
\item{"fixPointIter": }{number of fix-point iterations in the V-E step.
Default is 5 with no covariate, 2 otherwise.}
\item{"cores": }{integer for number of cores used. Default is 1.}
\item{"trace": }{integer for verbosity (0, 1, 2). Default is 1. Useless when \code{cores} > 1}
}

The different sampling designs are split into two families in which we find dyad-centered and
node-centered samplings. See \doi{10.1080/01621459.2018.1562934} for a complete description.
\itemize{
\item Missing at Random (MAR)
\itemize{
\item{"dyad": parameter = p = Prob(Dyad(i,j) is observed)}
\item{"node": parameter = p = Prob(Node i is observed)}
\item{"covar-dyad": parameter = beta in R^M, such that Prob(Dyad (i,j) is observed) = logistic(parameter' covarArray (i,j, .))}
\item{"covar-node": parameter = nu in R^M such that Prob(Node i is observed)  = logistic(parameter' covarMatrix (i,)}
\item{"snowball": parameter = number of waves with Prob(Node i is observed in the 1st wave)}
}
\item Not Missing At Random (NMAR)
\itemize{
\item{"double-standard": parameter = (p0,p1) with p0 = Prob(Dyad (i,j) is observed | the dyad is equal to 0), p1 = Prob(Dyad (i,j) is observed | the dyad is equal to 1)}
\item{"block-node": parameter = c(p(1),...,p(Q)) and p(q) = Prob(Node i is observed | node i is in cluster q)}
\item{"block-dyad": parameter = c(p(1,1),...,p(Q,Q)) and p(q,l) = Prob(Edge (i,j) is observed | node i is in cluster q and node j is in cluster l)}
\item{"degree": parameter = c(a,b) and logit(a+b*degree(i)) = Prob(Node i is observed | Degree(i))}
}
}
}
\examples{
## SBM parameters
N <- 300 # number of nodes
Q <- 3   # number of clusters
pi <- rep(1,Q)/Q     # block proportion
theta <- list(mean = diag(.45,Q) + .05 ) # connectivity matrix

## simulate an unidrected binary SBM without covariate
sbm <- sbm::sampleSimpleSBM(N, pi, theta)

## Sample network data

# some sampling design and their associated parameters
sampling_parameters <- list(
   "dyad" = .3,
   "node" = .3,
   "double-standard" = c(0.4, 0.8),
   "block-node" = c(.3, .8, .5),
   "block-dyad" = theta$mean,
   "degree" = c(.01, .01),
   "snowball" = c(2,.1)
 )

observed_networks <- list()

for (sampling in names(sampling_parameters)) {
  observed_networks[[sampling]] <-
     missSBM::observeNetwork(
       adjacencyMatrix = sbm$netMatrix,
       sampling        = sampling,
       parameters      = sampling_parameters[[sampling]],
       cluster         = sbm$memberships
     )
}
}
