% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/std.coef.R
\name{std.coef}
\alias{std.coef}
\title{Standardized Coefficients}
\usage{
std.coef(model, print = c("all", "stdx", "stdy", "stdyx"), digits = 3, p.digits = 4,
         check = TRUE, output = TRUE)
}
\arguments{
\item{model}{a fitted model of class \code{"lm"}.}

\item{print}{a character vector indicating which results to show, i.e. \code{"all"}, for all results,
\code{"stdx"} for standardizing only the predictor, \code{"stdy"} for for standardizing only
the criterion, and \code{"stdyx"} for for standardizing both the predictor and the criterion.
Note that the default setting is depending on the level of measurement of the predictors,
i.e., if all predictors are continuous, the default setting is \code{print = "stdyx"};
if all predictors are binary, the default setting is \code{print = "stdy"}; if predictors
are continuous and binary, the default setting is \code{print = c("stdy", "stdyx")}.}

\item{digits}{an integer value indicating the number of decimal places to be used for displaying
results.}

\item{p.digits}{an integer value indicating the number of decimal places to be used for displaying the
\emph{p}-value.}

\item{check}{logical: if \code{TRUE}, argument specification is checked.}

\item{output}{logical: if \code{TRUE}, output is shown on the console.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:
  \item{\code{call}}{function call}
  \item{\code{type}}{type of analysis}
  \item{\code{model}}{model specified in \code{model} }
  \item{\code{args}}{specification of function arguments}
  \item{\code{result}}{list with result tables, i.e., \code{coef} for the regression
                       table including standardized coefficients and \code{sd}
                       for the standard deviation of the outcome and predictor(s)}
}
\description{
This function computes standardized coefficients for linear models estimated by using the \code{lm()}
function.
}
\details{
The slope \eqn{\beta} can be standardized with respect to only \eqn{x}, only \eqn{y}, or both \eqn{y}
and \eqn{x}:

\deqn{StdX(\beta_1) = \beta_1 SD(x)}

\eqn{StdX(\beta_1)} standardizes with respect to \eqn{x} only and is interpreted as the change in
\eqn{y} when \eqn{x} changes one standard deviation referred to as \eqn{SD(x)}.

\deqn{StdY(\beta_1) = \frac{\beta_1}{SD(x)}}

\eqn{StdY(\beta_1)} standardizes with respect to \eqn{y} only and is interpreted as the change in
\eqn{y} standard deviation units, referred to as \eqn{SD(y)}, when \eqn{x} changes one unit.

\deqn{StdYX(\beta_1) = \beta_1 \frac{SD(x)}{SD(y)}}

\eqn{StdYX(\beta_1)} standardizes with respect to both \eqn{y} and \eqn{x} and is interpreted as the change
in \eqn{y} standard deviation units when \eqn{x} changes one standard deviation.

Note that the \eqn{StdYX(\beta_1)} and the \eqn{StdY(\beta_1)} standardizations are not suitable for the
slope of a binary predictor because a one standard deviation change in a binary variable is generally
not of interest (Muthen, Muthen, & Asparouhov, 2016).

The standardization of the slope \eqn{\beta_3} in a regression model with an interaction term uses the
product of standard deviations \eqn{SD(x_1)SD(x_2)} rather than the standard deviation of the product
\eqn{SD(x_1 x_2)} for the interaction variable \eqn{x_1}\eqn{x_2} (see Wen, Marsh & Hau, 2010). Likewise,
the standardization of the slope \eqn{\beta_3} in a polynomial regression model with a quadratic term
uses the product of standard deviations \eqn{SD(x)SD(x)} rather than the standard deviation of the
product \eqn{SD(x x)} for the quadratic term \eqn{x^2}.
}
\examples{
dat <- data.frame(x1 = c(3, 2, 4, 9, 5, 3, 6, 4, 5, 6, 3, 5),
                  x2 = c(1, 4, 3, 1, 2, 4, 3, 5, 1, 7, 8, 7),
                  x3 = c(0, 0, 1, 0, 1, 1, 1, 1, 0, 0, 1, 1),
                  y = c(2, 7, 4, 4, 7, 8, 4, 2, 5, 1, 3, 8))

#----------------------------
# Linear model

#...........
# Regression model with continuous predictors
mod.lm1 <- lm(y ~ x1 + x2, data = dat)
std.coef(mod.lm1)

# Print all standardized coefficients
std.coef(mod.lm1, print = "all")

#...........
# Regression model with dichotomous predictor
mod.lm2 <- lm(y ~ x3, data = dat)
std.coef(mod.lm2)

#...........
# Regression model with continuous and dichotomous predictors
mod.lm3 <- lm(y ~ x1 + x2 + x3, data = dat)
std.coef(mod.lm3)

#...........
# Regression model with continuous predictors and an interaction term
mod.lm4 <- lm(y ~ x1*x2, data = dat)

#...........
# Regression model with a quadratic term
mod.lm5 <- lm(y ~ x1 + I(x1^2), data = dat)
std.coef(mod.lm5)
}
\references{
Muthen, B. O., Muthen, L. K., & Asparouhov, T. (2016). \emph{Regression and mediation analysis using Mplus}.
Muthen & Muthen.

Wen, Z., Marsh, H. W., & Hau, K.-T. (2010). Structural equation models of latent interactions:
An appropriate standardized solution and its scale-free properties. \emph{Structural Equation Modeling:
A Multidisciplinary Journal, 17}, 1-22. https://doi.org/10.1080/10705510903438872
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
