% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summa.R
\name{summa}
\alias{summa}
\title{Print Summary Output}
\usage{
summa(model, print = c("all", "default", "call", "descript", "cormat", "modsum",
                       "coef", "confint", "stdcoef", "vif"),
      robust = FALSE, ddf = c("Satterthwaite", "Kenward-Roger", "lme4"),
      conf.level = 0.95, method = c("profile", "wald", "boot"), R = 1000,
      boot = c("perc", "basic", "norm"), seed = NULL, digits = 2, p.digits = 3,
      write = NULL, append = TRUE, check = TRUE, output = TRUE)
}
\arguments{
\item{model}{a fitted model of class \code{"lm"}, \code{"lmerMod"}, or
\code{"lmerModLmerTest"}}

\item{print}{a character vector indicating which results to print, i.e.
\code{"all"}, for all results, \code{"call"} for the function
call, \code{"descript"} for descriptive statistics,
\code{cormat} for the Pearson product-moment correlation
matrix for models estimated by \code{"lm"} (see
\code{\link{cor.matrix}} function) or within- and between-group
correlation matrix for models estimated by \code{"lmer"}
(see \code{\link{multilevel.cor}} function), \code{modsum}
for the multiple correlation, r-squared, and F-test for models
estimated by \code{"lm"} or model summary, marginal, and
conditional R-squared for models estimated by \code{"lmer"}
(see \code{\link{multilevel.r2}} function), \code{coef} for the
unstandardized coefficients for models estimated by \code{"lm"}
or random effects and fixed effects for models estimated
by \code{"lmer"}, \code{confint} for the confidence interval
for unstandardized coefficients, \code{stdcoef} for the
standardized coefficients (see \code{\link{coeff.std} function}),
and \code{vif} for the variance inflation factor (see
\code{\link{check.collin}} function). The default setting is
\code{print = c("call", "modsum", "coef")}.}

\item{robust}{logical: if \code{TRUE}, heteroscedasticity-consistent
standard errors and heteroscedasticity-robust F-test using
the HC4 estimator is computed for linear models estimated
by using the \code{lm()} function (see \code{\link{coeff.robust}}
function).}

\item{ddf}{a character string for specifying the method for computing
the degrees of freedom when using the \pkg{lmerTest} package
to obtain \emph{p}-values for fixed effects in multilevel
and linear mixed-effects models, i.e., \code{"Satterthwaite"}
(default) for Satterthwaite's method, \code{"Kenward-Roger"}
for the Kenward-Roger's method, and \code{"lme4"} for the
lme4-summary without degrees of freedom and significance
values (see Kuznetsova et al., 2017).}

\item{conf.level}{a numeric value between 0 and 1 indicating the confidence
level of the interval.}

\item{method}{a character string for specifying the method for computing
confidence intervals (CI), i.e., \code{"profile"} (default)
for computing a likelihood profile and finding the appropriate
cutoffs based on the likelihood ratio test, \code{"Wald"}
for approximating the CIs based on the estimated local curvature
of the likelihood surface, and \code{"boot"} for performing
bootstrapping with CIs computed from the bootstrap distribution
according to the argument \code{boot}.}

\item{R}{a numeric value indicating the number of bootstrap replicates
(default is 1000).}

\item{boot}{a character string for specifying the type of bootstrap
confidence intervals (CI), i.e., i.e., \code{"perc"} (default),
for the percentile bootstrap CI, \code{"basic"} for the basic
bootstrap CI, and \code{"norm"} for the normal approximation
bootstrap CI.}

\item{seed}{a numeric value specifying seeds of the pseudo-random
numbers used in the bootstrap algorithm when conducting
bootstrapping.}

\item{digits}{an integer value indicating the number of decimal places
to be used.}

\item{p.digits}{an integer value indicating the number of decimal places
to be used for displaying multiple R, R-squared and
\emph{p}-value.}

\item{write}{a character string naming a file for writing the output into
either a text file with file extension \code{".txt"} (e.g.,
\code{"Output.txt"}) or Excel file with file extension
\code{".xlsx"}  (e.g., \code{"Output.xlsx"}). If the file
name does not contain any file extension, an Excel file will
be written.}

\item{append}{logical: if \code{TRUE} (default), output will be appended
to an existing text file with extension \code{.txt} specified
in \code{write}, if \code{FALSE} existing text file will be
overwritten.}

\item{check}{logical: if \code{TRUE} (default), argument specification
is checked.}

\item{output}{logical: if \code{TRUE} (default), output is shown on the console.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:
\item{\code{call}}{function call}
\item{\code{type}}{type of analysis}
\item{\code{model}}{model specified in \code{model}}
\item{\code{args}}{specification of function arguments}
\item{\code{result}}{list with results, i.e., \code{call} for the the function
call, \code{call} for descriptive statistics, \code{cormat} for the
correlation matrix, \code{modsum} for the model summary, \code{randeff} for
the variance and correlation components,\code{coef} for the model coefficients,
and \code{converg} for the convergence check, i.e., \code{1} = model converged,
\code{0} = model singular, and \code{-1} model not converged.}
}
\description{
This function prints a summary of the result object returned by the function
\code{"lm"} for estimating linear regression models and for the result object
returned by the function \code{"lmer"} from the \pkg{lme4} package to estimate
two- or three-level multilevel and linear mixed-effects models. By default,
the function prints the function call, model summary, and the regression
coefficient table.
}
\examples{
#----------------------------------------------------------------------------
# Linear Model

# Estimate linear model
mod.lm <- lm(mpg ~ cyl + disp, data = mtcars)

# Example 1a: Default setting
summa(mod.lm)

# Example 1b: Heteroscedasticity-consistent standard errors
summa(mod.lm, robust = TRUE)

# Example 1c: Print all available results
summa(mod.lm, print = "all")

# Example 1d: Print default results plus standardized coefficient
summa(mod.lm, print = c("default", "stdcoef"))

\dontrun{
#----------------------------------------------------------------------------
# Multilevel and Linear Mixed-Effects Model

# Load lme4 package
library(lme4)

# Load data set "Demo.twolevel" in the lavaan package
data("Demo.twolevel", package = "lavaan")

#------------------
## Two-Level Data

# Cluster-mean centering, center() from the misty package
Demo.twolevel <- center(Demo.twolevel, x2, type = "CWC", cluster = "cluster")

# Grand-mean centering, center() from the misty package
Demo.twolevel <- center(Demo.twolevel, w1, type = "CGM", cluster = "cluster")

# Estimate two-level mixed-effects model
mod.lmer2 <- lmer(y1 ~ x2.c + w1.c + x2.c:w1.c + (1 + x2.c | cluster),
                  data = Demo.twolevel)

# Example 2a: Default setting
summa(mod.lmer2)

# Example 2b: Print all available results
summa(mod.lmer2, print = "all")

# Example 2c: Print default results plus standardized coefficient
summa(mod.lmer2, print = c("default", "stdcoef"))

# Load lmerTest package
library(lmerTest)

# Re-estimate two-level model using the lme4 and lmerTest package
mod.lmer2 <- lmer(y1 ~ x2.c + w1.c + x2.c:w1.c + (1 + x2.c | cluster), data = Demo.twolevel)

# Example 2d: Default setting, Satterthwaite's method
summa(mod.lmer2)

# Example 2e: Kenward-Roger's method
summa(mod.lmer2, ddf = "Kenward-Roger")

#------------------
## Three-Level Data

# Create arbitrary three-level data
Demo.threelevel <- data.frame(Demo.twolevel, cluster2 = Demo.twolevel$cluster,
                                             cluster3 = rep(1:10, each = 250))

# Cluster-mean centering, center() from the misty package
Demo.threelevel <- center(Demo.threelevel, x1, type = "CWC",
                          cluster = c("cluster3", "cluster2"))

# Cluster-mean centering, center() from the misty package
Demo.threelevel <- center(Demo.threelevel, w1, type = "CWC",
                          cluster = c("cluster3", "cluster2"))

# Estimate three-level mixed-effects model
mod.lmer3 <- lmer(y1 ~ x1.c + w1.c + (1 | cluster3/cluster2),
                  data = Demo.threelevel)

# Example 3a: Default setting
summa(mod.lmer3)

# Example 3b: DPrint all available results
summa(mod.lmer3, print = "all")

#----------------------------------------------------------------------------
# Write Results

# Example 4a: Write Results into a text file
summa(mod.lm, print = "all", write = "Linear_Model.txt")

# Example 4b: Write Results into a Excel file
summa(mod.lm, print = "all", write = "Linear_Model.xlsx")
}
}
\references{
Kuznetsova, A, Brockhoff, P. B., & Christensen, R. H. B. (2017). lmerTest Package:
Tests in linear mixed effects models. \emph{Journal of Statistical Software, 82}
13, 1-26. https://doi.org/10.18637/jss.v082.i13.
}
\seealso{
\code{\link{descript}}, \code{\link{cor.matrix}}, \code{\link{coeff.std}},
\code{\link{coeff.robust}}, \code{\link{check.collin}}
}
\author{
Takuya Yanagida
}
