% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Learner.R
\name{Learner}
\alias{Learner}
\title{Learner Class}
\description{
This is the abstract base class for learner objects like \link{LearnerClassif} and \link{LearnerRegr}.

Learners are build around the three following key parts:
\itemize{
\item Methods \verb{$train()} and \verb{$predict()} which call internal methods or private methods \verb{$.train()}/\verb{$.predict()}).
\item A \link[paradox:ParamSet]{paradox::ParamSet} which stores meta-information about available hyperparameters, and also stores hyperparameter settings.
\item Meta-information about the requirements and capabilities of the learner.
\item The fitted model stored in field \verb{$model}, available after calling \verb{$train()}.
}

Predefined learners are stored in the \link[mlr3misc:Dictionary]{dictionary} \link{mlr_learners},
e.g. \code{\link[=mlr_learners_classif.rpart]{classif.rpart}} or \code{\link[=mlr_learners_regr.rpart]{regr.rpart}}.

More classification and regression learners are implemented in the add-on package \CRANpkg{mlr3learners}.
Learners for survival analysis (or more general, for probabilistic regression) can be found in \CRANpkg{mlr3proba}.
Unsupervised cluster algorithms are implemented in \CRANpkg{mlr3cluster}.
The dictionary \link{mlr_learners} gets automatically populated with the new learners as soon as the respective packages are loaded.

More (experimental) learners can be found in the GitHub repository: \url{https://github.com/mlr-org/mlr3extralearners}.
A guide on how to extend \CRANpkg{mlr3} with custom learners can be found in the \href{https://mlr3book.mlr-org.com}{mlr3book}.

To combine the learner with preprocessing operations like factor encoding, \CRANpkg{mlr3pipelines} is recommended.
Hyperparameters stored in the \code{param_set} can be tuned with \CRANpkg{mlr3tuning}.
}
\section{Optional Extractors}{


Specific learner implementations are free to implement additional getters to ease the access of certain parts
of the model in the inherited subclasses.

For the following operations, extractors are standardized:
\itemize{
\item \code{importance(...)}: Returns the feature importance score as numeric vector.
The higher the score, the more important the variable.
The returned vector is named with feature names and sorted in decreasing order.
Note that the model might omit features it has not used at all.
The learner must be tagged with property \code{"importance"}.
To filter variables using the importance scores, see package \CRANpkg{mlr3filters}.
\item \code{selected_features(...)}: Returns a subset of selected features as \code{character()}.
The learner must be tagged with property \code{"selected_features"}.
\item \code{oob_error(...)}: Returns the out-of-bag error of the model as \code{numeric(1)}.
The learner must be tagged with property \code{"oob_error"}.
\item \code{internal_valid_scores}: Returns the internal validation score(s) of the model as a named \code{list()}.
Only available for \code{\link{Learner}}s with the \code{"validation"} property.
If the learner is not trained yet, this returns \code{NULL}.
\item \code{internal_tuned_values}: Returns the internally tuned hyperparameters of the model as a named \code{list()}.
Only available for \code{\link{Learner}}s with the \code{"internal_tuning"} property.
If the learner is not trained yet, this returns \code{NULL}.
}
}

\section{Setting Hyperparameters}{


All information about hyperparameters is stored in the slot \code{param_set} which is a \link[paradox:ParamSet]{paradox::ParamSet}.
The printer gives an overview about the ids of available hyperparameters, their storage type, lower and upper bounds,
possible levels (for factors), default values and assigned values.
To set hyperparameters, assign a named list to the subslot \code{values}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{lrn = lrn("classif.rpart")
lrn$param_set$values = list(minsplit = 3, cp = 0.01)
}\if{html}{\out{</div>}}

Note that this operation replaces all previously set hyperparameter values.
If you only intend to change one specific hyperparameter value and leave the others as-is, you can use the helper function \code{\link[mlr3misc:insert_named]{mlr3misc::insert_named()}}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{lrn$param_set$values = mlr3misc::insert_named(lrn$param_set$values, list(cp = 0.001))
}\if{html}{\out{</div>}}

If the learner has additional hyperparameters which are not encoded in the \link[paradox:ParamSet]{ParamSet}, you can easily extend the learner.
Here, we add a factor hyperparameter with id \code{"foo"} and possible levels \code{"a"} and \code{"b"}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{lrn$param_set$add(paradox::ParamFct$new("foo", levels = c("a", "b")))
}\if{html}{\out{</div>}}
}

\section{Implementing Validation}{

Some Learners, such as \code{XGBoost}, other boosting algorithms, or deep learning models (\code{mlr3torch}),
utilize validation data during the training to prevent overfitting or to log the validation performance.
It is possible to configure learners to be able to receive such an independent validation set during training.
To do so, one must:
\itemize{
\item annotate the learner with the \code{"validation"} property
\item implement the active binding \verb{$internal_valid_scores} (see section \emph{Optional Extractors}), as well as the
private method \verb{$.extract_internal_valid_scores()} which returns the (final) internal validation scores from the
model of the \code{\link{Learner}} and returns them as a named \code{list()} of \code{numeric(1)}.
If the model is not trained yet, this method should return \code{NULL}.
\item Add the \code{validate} parameter, which can be either \code{NULL}, a ratio in $(0, 1)$, \code{"test"}, or \code{"predefined"}:
\itemize{
\item \code{NULL}: no validation
\item \code{ratio}: only proportion \code{1 - ratio} of the task is used for training and \code{ratio} is used for validation.
\item \code{"test"} means that the \code{"test"} task is used.
\strong{Warning}: This can lead to biased performance estimation.
This option is only available if the learner is being trained via \code{\link[=resample]{resample()}}, \code{\link[=benchmark]{benchmark()}} or functions that
internally use them, e.g. \code{tune()} of \CRANpkg{mlr3tuning} or \code{batchmark()} of \CRANpkg{mlr3batchmark}.
This is especially useful for hyperparameter tuning, where one might e.g. want to use the same validation data
for early stopping and model evaluation.
\item \code{"predefined"} means that the task's (manually set) \verb{$internal_valid_task} is used.
See the \code{\link{Task}} documentation for more information.
}
}

For an example how to do this, see \code{\link{LearnerClassifDebug}}.
Note that in \code{.train()}, the \verb{$internal_valid_task} will only be present if the \verb{$validate} field of the \code{Learner}
is set to a non-\code{NULL} value.
}

\section{Implementing Internal Tuning}{

Some learners such as \code{XGBoost} or \code{cv.glmnet} can internally tune hyperparameters.
XGBoost, for example, can tune the number of boosting rounds based on the validation performance.
CV Glmnet, on the other hand, can tune the regularization parameter based on an internal cross-validation.
Internal tuning \emph{can} therefore rely on the internal validation data, but does not necessarily do so.

In order to be able to combine this internal hyperparamer tuning with the standard hyperparameter optimization
implemented via \CRANpkg{mlr3tuning}, one most:
\itemize{
\item annotate the learner with the \code{"internal_tuning"} property
\item implement the active binding \verb{$internal_tuned_values} (see section \emph{Optional Extractors}) as well as the
private method \verb{$.extract_internal_tuned_values()} which extracts the internally tuned values from the \code{\link{Learner}}'s
model and returns them as a named \code{list()}.
If the model is not trained yet, this method should return \code{NULL}.
\item Have at least one parameter tagged with \code{"internal_tuning"}, which requires to also provide a \code{in_tune_fn} and
\code{disable_tune_fn}, and \emph{should} also include a default \code{aggr}egation function.
}

For an example how to do this, see \code{\link{LearnerClassifDebug}}.
}

\section{Implementing Marshaling}{

Some \code{\link{Learner}}s have models that cannot be serialized as they e.g. contain external pointers.
In order to still be able to save them, use them with parallelization or callr encapsulation it is necessary
to implement how they should be (un)-marshaled. See \code{\link{marshaling}} for how to do this.
}

\seealso{
\itemize{
\item Chapter in the \href{https://mlr3book.mlr-org.com/}{mlr3book}:
\url{https://mlr3book.mlr-org.com/chapters/chapter2/data_and_basic_modeling.html#sec-learners}
\item Package \CRANpkg{mlr3learners} for a solid collection of essential learners.
\item Package \href{https://github.com/mlr-org/mlr3extralearners}{mlr3extralearners} for more learners.
\item \link[mlr3misc:Dictionary]{Dictionary} of \link[=Learner]{Learners}: \link{mlr_learners}
\item \code{as.data.table(mlr_learners)} for a table of available \link[=Learner]{Learners} in the running session (depending on the loaded packages).
\item \CRANpkg{mlr3pipelines} to combine learners with pre- and postprocessing steps.
\item Package \CRANpkg{mlr3viz} for some generic visualizations.
\item Extension packages for additional task types:
\itemize{
\item \CRANpkg{mlr3proba} for probabilistic supervised regression and survival analysis.
\item \CRANpkg{mlr3cluster} for unsupervised clustering.
}
\item \CRANpkg{mlr3tuning} for tuning of hyperparameters, \CRANpkg{mlr3tuningspaces}
for established default tuning spaces.
}

Other Learner: 
\code{\link{LearnerClassif}},
\code{\link{LearnerRegr}},
\code{\link{mlr_learners}},
\code{\link{mlr_learners_classif.debug}},
\code{\link{mlr_learners_classif.featureless}},
\code{\link{mlr_learners_classif.rpart}},
\code{\link{mlr_learners_regr.debug}},
\code{\link{mlr_learners_regr.featureless}},
\code{\link{mlr_learners_regr.rpart}}
}
\concept{Learner}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{id}}{(\code{character(1)})\cr
Identifier of the object.
Used in tables, plot and text output.}

\item{\code{label}}{(\code{character(1)})\cr
Label for this object.
Can be used in tables, plot and text output instead of the ID.}

\item{\code{state}}{(\code{NULL} | named \code{list()})\cr
Current (internal) state of the learner.
Contains all information gathered during \code{train()} and \code{predict()}.
It is not recommended to access elements from \code{state} directly.
This is an internal data structure which may change in the future.}

\item{\code{task_type}}{(\code{character(1)})\cr
Task type, e.g. \code{"classif"} or \code{"regr"}.

For a complete list of possible task types (depending on the loaded packages),
see \code{\link[=mlr_reflections]{mlr_reflections$task_types$type}}.}

\item{\code{predict_types}}{(\code{character()})\cr
Stores the possible predict types the learner is capable of.
A complete list of candidate predict types, grouped by task type, is stored in \code{\link[=mlr_reflections]{mlr_reflections$learner_predict_types}}.}

\item{\code{feature_types}}{(\code{character()})\cr
Stores the feature types the learner can handle, e.g. \code{"logical"}, \code{"numeric"}, or \code{"factor"}.
A complete list of candidate feature types, grouped by task type, is stored in \code{\link[=mlr_reflections]{mlr_reflections$task_feature_types}}.}

\item{\code{properties}}{(\code{character()})\cr
Stores a set of properties/capabilities the learner has.
A complete list of candidate properties, grouped by task type, is stored in \code{\link[=mlr_reflections]{mlr_reflections$learner_properties}}.}

\item{\code{packages}}{(\code{character(1)})\cr
Set of required packages.
These packages are loaded, but not attached.}

\item{\code{predict_sets}}{(\code{character()})\cr
During \code{\link[=resample]{resample()}}/\code{\link[=benchmark]{benchmark()}}, a \link{Learner} can predict on multiple sets.
Per default, a learner only predicts observations in the test set (\code{predict_sets == "test"}).
To change this behavior, set \code{predict_sets} to a non-empty subset of \verb{\{"train", "test", "internal_valid"\}}.
The \code{"train"} predict set contains the train ids from the resampling. This means that if a learner does validation and
sets \verb{$validate} to a ratio (creating the validation data from the training data), the train predictions
will include the predictions for the validation data.
Each set yields a separate \link{Prediction} object.
Those can be combined via getters in \link{ResampleResult}/\link{BenchmarkResult}, or \link{Measure}s can be configured
to operate on specific subsets of the calculated prediction sets.}

\item{\code{parallel_predict}}{(\code{logical(1)})\cr
If set to \code{TRUE}, use \CRANpkg{future} to calculate predictions in parallel (default: \code{FALSE}).
The row ids of the \code{task} will be split into \code{\link[future:nbrOfWorkers]{future::nbrOfWorkers()}} chunks,
and predictions are evaluated according to the active \code{\link[future:plan]{future::plan()}}.
This currently only works for methods \code{Learner$predict()} and \code{Learner$predict_newdata()},
and has no effect during \code{\link[=resample]{resample()}} or \code{\link[=benchmark]{benchmark()}} where you have other means
to parallelize.

Note that the recorded time required for prediction reports the time required to predict
is not properly defined and depends on the parallelization backend.}

\item{\code{timeout}}{(named \code{numeric(2)})\cr
Timeout for the learner's train and predict steps, in seconds.
This works differently for different encapsulation methods, see
\code{\link[mlr3misc:encapsulate]{mlr3misc::encapsulate()}}.
Default is \code{c(train = Inf, predict = Inf)}.
Also see the section on error handling the mlr3book:
\url{https://mlr3book.mlr-org.com/chapters/chapter10/advanced_technical_aspects_of_mlr3.html#sec-error-handling}}

\item{\code{man}}{(\code{character(1)})\cr
String in the format \verb{[pkg]::[topic]} pointing to a manual page for this object.
Defaults to \code{NA}, but can be set by child classes.}
}
\if{html}{\out{</div>}}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{data_formats}}{(\code{character()})\cr
Supported data format. Always \code{"data.table"}..
This is deprecated and will be removed in the future.}

\item{\code{model}}{(any)\cr
The fitted model. Only available after \verb{$train()} has been called.}

\item{\code{timings}}{(named \code{numeric(2)})\cr
Elapsed time in seconds for the steps \code{"train"} and \code{"predict"}.

When predictions for multiple predict sets were made during \code{\link[=resample]{resample()}} or \code{\link[=benchmark]{benchmark()}},
the predict time shows the cumulative duration of all predictions.
If \code{learner$predict()} is called manually, the last predict time gets overwritten.

Measured via \code{\link[mlr3misc:encapsulate]{mlr3misc::encapsulate()}}.}

\item{\code{log}}{(\code{\link[data.table:data.table]{data.table::data.table()}})\cr
Returns the output (including warning and errors) as table with columns
\itemize{
\item \code{"stage"} ("train" or "predict"),
\item \code{"class"} ("output", "warning", or "error"), and
\item \code{"msg"} (\code{character()}).
}}

\item{\code{warnings}}{(\code{character()})\cr
Logged warnings as vector.}

\item{\code{errors}}{(\code{character()})\cr
Logged errors as vector.}

\item{\code{hash}}{(\code{character(1)})\cr
Hash (unique identifier) for this object.
The hash is calculated based on the learner id, the parameter settings, the predict type, the fallback hash, the parallel predict setting, the validate setting, and the predict sets.}

\item{\code{phash}}{(\code{character(1)})\cr
Hash (unique identifier) for this partial object, excluding some components which are varied systematically during tuning (parameter values).}

\item{\code{predict_type}}{(\code{character(1)})\cr
Stores the currently active predict type, e.g. \code{"response"}.
Must be an element of \verb{$predict_types}.}

\item{\code{param_set}}{(\link[paradox:ParamSet]{paradox::ParamSet})\cr
Set of hyperparameters.}

\item{\code{fallback}}{(\link{Learner})\cr
Returns the fallback learner set with \verb{$encapsulate()}.}

\item{\code{encapsulation}}{(\code{character(2)})\cr
Returns the encapsulation settings set with \verb{$encapsulate()}.}

\item{\code{hotstart_stack}}{(\link{HotstartStack})\cr.
Stores \code{HotstartStack}.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Learner-new}{\code{Learner$new()}}
\item \href{#method-Learner-format}{\code{Learner$format()}}
\item \href{#method-Learner-print}{\code{Learner$print()}}
\item \href{#method-Learner-help}{\code{Learner$help()}}
\item \href{#method-Learner-train}{\code{Learner$train()}}
\item \href{#method-Learner-predict}{\code{Learner$predict()}}
\item \href{#method-Learner-predict_newdata}{\code{Learner$predict_newdata()}}
\item \href{#method-Learner-reset}{\code{Learner$reset()}}
\item \href{#method-Learner-base_learner}{\code{Learner$base_learner()}}
\item \href{#method-Learner-encapsulate}{\code{Learner$encapsulate()}}
\item \href{#method-Learner-clone}{\code{Learner$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-new"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.

Note that this object is typically constructed via a derived classes, e.g. \link{LearnerClassif} or \link{LearnerRegr}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$new(
  id,
  task_type,
  param_set = ps(),
  predict_types = character(),
  feature_types = character(),
  properties = character(),
  data_formats,
  packages = character(),
  label = NA_character_,
  man = NA_character_
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{id}}{(\code{character(1)})\cr
Identifier for the new instance.}

\item{\code{task_type}}{(\code{character(1)})\cr
Type of task, e.g. \code{"regr"} or \code{"classif"}.
Must be an element of \link[=mlr_reflections]{mlr_reflections$task_types$type}.}

\item{\code{param_set}}{(\link[paradox:ParamSet]{paradox::ParamSet})\cr
Set of hyperparameters.}

\item{\code{predict_types}}{(\code{character()})\cr
Supported predict types. Must be a subset of \code{\link[=mlr_reflections]{mlr_reflections$learner_predict_types}}.}

\item{\code{feature_types}}{(\code{character()})\cr
Feature types the learner operates on. Must be a subset of \code{\link[=mlr_reflections]{mlr_reflections$task_feature_types}}.}

\item{\code{properties}}{(\code{character()})\cr
Set of properties of the \link{Learner}.
Must be a subset of \code{\link[=mlr_reflections]{mlr_reflections$learner_properties}}.
The following properties are currently standardized and understood by learners in \CRANpkg{mlr3}:
\itemize{
\item \code{"missings"}: The learner can handle missing values in the data.
\item \code{"weights"}: The learner supports observation weights.
\item \code{"importance"}: The learner supports extraction of importance scores, i.e. comes with an \verb{$importance()} extractor function (see section on optional extractors in \link{Learner}).
\item \code{"selected_features"}: The learner supports extraction of the set of selected features, i.e. comes with a \verb{$selected_features()} extractor function (see section on optional extractors in \link{Learner}).
\item \code{"oob_error"}: The learner supports extraction of estimated out of bag error, i.e. comes with a \code{oob_error()} extractor function (see section on optional extractors in \link{Learner}).
\item \code{"validation"}: The learner can use a validation task during training.
\item \code{"internal_tuning"}: The learner is able to internally optimize hyperparameters (those are also tagged with \code{"internal_tuning"}).
\item \code{"marshal"}: To save learners with this property, you need to call \verb{$marshal()} first.
If a learner is in a marshaled state, you call first need to call \verb{$unmarshal()} to use its model, e.g. for prediction.
}}

\item{\code{data_formats}}{(\code{character()})\cr
Deprecated: ignored, and will be removed in the future.}

\item{\code{packages}}{(\code{character()})\cr
Set of required packages.
A warning is signaled by the constructor if at least one of the packages is not installed,
but loaded (not attached) later on-demand via \code{\link[=requireNamespace]{requireNamespace()}}.}

\item{\code{label}}{(\code{character(1)})\cr
Label for the new instance.}

\item{\code{man}}{(\code{character(1)})\cr
String in the format \verb{[pkg]::[topic]} pointing to a manual page for this object.
The referenced help package can be opened via method \verb{$help()}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-format"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-format}{}}}
\subsection{Method \code{format()}}{
Helper for print outputs.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$format(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{(ignored).}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-print"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-print}{}}}
\subsection{Method \code{print()}}{
Printer.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$print(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{(ignored).}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-help"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-help}{}}}
\subsection{Method \code{help()}}{
Opens the corresponding help page referenced by field \verb{$man}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$help()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-train"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-train}{}}}
\subsection{Method \code{train()}}{
Train the learner on a set of observations of the provided \code{task}.
Mutates the learner by reference, i.e. stores the model alongside other information in field \verb{$state}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$train(task, row_ids = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{task}}{(\link{Task}).}

\item{\code{row_ids}}{(\code{integer()})\cr
Vector of training indices as subset of \code{task$row_ids}.
For a simple split into training and test set, see \code{\link[=partition]{partition()}}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Returns the object itself, but modified \strong{by reference}.
You need to explicitly \verb{$clone()} the object beforehand if you want to keeps
the object in its previous state.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-predict"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-predict}{}}}
\subsection{Method \code{predict()}}{
Uses the information stored during \verb{$train()} in \verb{$state} to create a new \link{Prediction}
for a set of observations of the provided \code{task}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$predict(task, row_ids = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{task}}{(\link{Task}).}

\item{\code{row_ids}}{(\code{integer()})\cr
Vector of test indices as subset of \code{task$row_ids}.
For a simple split into training and test set, see \code{\link[=partition]{partition()}}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\link{Prediction}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-predict_newdata"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-predict_newdata}{}}}
\subsection{Method \code{predict_newdata()}}{
Uses the model fitted during \verb{$train()} to create a new \link{Prediction} based on the new data in \code{newdata}.
Object \code{task} is the task used during \verb{$train()} and required for conversion of \code{newdata}.
If the learner's \verb{$train()} method has been called, there is a (size reduced) version
of the training task stored in the learner.
If the learner has been fitted via \code{\link[=resample]{resample()}} or \code{\link[=benchmark]{benchmark()}}, you need to pass the corresponding task stored
in the \link{ResampleResult} or \link{BenchmarkResult}, respectively.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$predict_newdata(newdata, task = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{newdata}}{(any object supported by \code{\link[=as_data_backend]{as_data_backend()}})\cr
New data to predict on.
All data formats convertible by \code{\link[=as_data_backend]{as_data_backend()}} are supported, e.g.
\code{data.frame()} or \link{DataBackend}.
If a \link{DataBackend} is provided as \code{newdata}, the row ids are preserved,
otherwise they are set to to the sequence \code{1:nrow(newdata)}.}

\item{\code{task}}{(\link{Task}).}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\link{Prediction}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-reset"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-reset}{}}}
\subsection{Method \code{reset()}}{
Reset the learner, i.e. un-train by resetting the \code{state}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$reset()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
Returns the object itself, but modified \strong{by reference}.
You need to explicitly \verb{$clone()} the object beforehand if you want to keeps
the object in its previous state.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-base_learner"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-base_learner}{}}}
\subsection{Method \code{base_learner()}}{
Extracts the base learner from nested learner objects like
\code{GraphLearner} in \CRANpkg{mlr3pipelines} or \code{AutoTuner} in
\CRANpkg{mlr3tuning}.
Returns the \link{Learner} itself for regular learners.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$base_learner(recursive = Inf)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{recursive}}{(\code{integer(1)})\cr
Depth of recursion for multiple nested objects.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\link{Learner}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-encapsulate"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-encapsulate}{}}}
\subsection{Method \code{encapsulate()}}{
Sets the encapsulation method and fallback learner for the train and predict steps.
There are currently four different methods implemented:
\itemize{
\item \code{"none"}: Just runs the learner in the current session and measures the elapsed time.
Does not keep a log, output is printed directly to the console.
Works well together with \code{\link[=traceback]{traceback()}}.
\item \code{"try"}: Similar to \code{"none"}, but catches error.
Output is printed to the console and not logged.
\item \code{"evaluate"}: Uses the package \CRANpkg{evaluate} to call the learner, measure time and do the logging.
\item \code{"callr"}: Uses the package \CRANpkg{callr} to call the learner, measure time and do the logging.
This encapsulation spawns a separate R session in which the learner is called.
While this comes with a considerable overhead, it also guards your session from being teared down by segfaults.
}

The fallback learner is fitted to create valid predictions in case that either the model fitting or the prediction of the original learner fails.
If the training step or the predict step of the original learner fails, the fallback is used completely to predict predictions sets.
If the original learner only partially fails during predict step (usually in the form of missing to predict some observations or producing some \verb{NA`` predictions), these missing predictions are imputed by the fallback. Note that the fallback is always trained, as we do not know in advance whether prediction will fail. If the training step fails, the }$model\verb{field of the original learner is}NULL`.

Also see the section on error handling the mlr3book:
\url{https://mlr3book.mlr-org.com/chapters/chapter10/advanced_technical_aspects_of_mlr3.html#sec-error-handling}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$encapsulate(method, fallback = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{method}}{\code{character(1)}\cr
One of \code{"none"}, \code{"try"}, \code{"evaluate"} or \code{"callr"}.
See the description for details.}

\item{\code{fallback}}{\link{Learner}\cr
The fallback learner for failed predictions.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\code{self} (invisibly).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Learner-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Learner-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Learner$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
