% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TuningInstanceBatchMulticrit.R
\name{TuningInstanceBatchMultiCrit}
\alias{TuningInstanceBatchMultiCrit}
\title{Class for Multi Criteria Tuning}
\description{
The \link{TuningInstanceBatchMultiCrit} specifies a tuning problem for a \link{Tuner}.
The function \code{\link[=ti]{ti()}} creates a \link{TuningInstanceBatchMultiCrit} and the function \code{\link[=tune]{tune()}} creates an instance internally.
}
\details{
The instance contains an \link{ObjectiveTuningBatch} object that encodes the black box objective function a \link{Tuner} has to optimize.
The instance allows the basic operations of querying the objective at design points (\verb{$eval_batch()}).
This operation is usually done by the \link{Tuner}.
Evaluations of hyperparameter configurations are performed in batches by calling \code{\link[mlr3:benchmark]{mlr3::benchmark()}} internally.
The evaluated hyperparameter configurations are stored in the \link{ArchiveBatchTuning} (\verb{$archive}).
Before a batch is evaluated, the \link[bbotk:Terminator]{bbotk::Terminator} is queried for the remaining budget.
If the available budget is exhausted, an exception is raised, and no further evaluations can be performed from this point on.
The tuner is also supposed to store its final result, consisting of a selected hyperparameter configuration and associated estimated performance values, by calling the method \code{instance$assign_result}.
}
\section{Resources}{

There are several sections about hyperparameter optimization in the \href{https://mlr3book.mlr-org.com}{mlr3book}.
\itemize{
\item Learn about \href{https://mlr3book.mlr-org.com/chapters/chapter5/advanced_tuning_methods_and_black_box_optimization.html#sec-multi-metrics-tuning}{multi-objective optimization}.
}

The \href{https://mlr-org.com/gallery-all-optimization.html}{gallery} features a collection of case studies and demos about optimization.
}

\section{Analysis}{

For analyzing the tuning results, it is recommended to pass the \link{ArchiveBatchTuning} to \code{as.data.table()}.
The returned data table is joined with the benchmark result which adds the \link[mlr3:ResampleResult]{mlr3::ResampleResult} for each hyperparameter evaluation.

The archive provides various getters (e.g. \verb{$learners()}) to ease the access.
All getters extract by position (\code{i}) or unique hash (\code{uhash}).
For a complete list of all getters see the methods section.

The benchmark result (\verb{$benchmark_result}) allows to score the hyperparameter configurations again on a different measure.
Alternatively, measures can be supplied to \code{as.data.table()}.

The \CRANpkg{mlr3viz} package provides visualizations for tuning results.
}

\examples{
# Hyperparameter optimization on the Palmer Penguins data set
task = tsk("penguins")

# Load learner and set search space
learner = lrn("classif.rpart",
  cp = to_tune(1e-04, 1e-1, logscale = TRUE)
)

# Construct tuning instance
instance = ti(
  task = task,
  learner = learner,
  resampling = rsmp("cv", folds = 3),
  measures = msrs(c("classif.ce", "time_train")),
  terminator = trm("evals", n_evals = 4)
)

# Choose optimization algorithm
tuner = tnr("random_search", batch_size = 2)

# Run tuning
tuner$optimize(instance)

# Optimal hyperparameter configurations
instance$result

# Inspect all evaluated configurations
as.data.table(instance$archive)
}
\section{Super classes}{
\code{\link[bbotk:OptimInstance]{bbotk::OptimInstance}} -> \code{\link[bbotk:OptimInstanceBatch]{bbotk::OptimInstanceBatch}} -> \code{\link[bbotk:OptimInstanceBatchMultiCrit]{bbotk::OptimInstanceBatchMultiCrit}} -> \code{TuningInstanceBatchMultiCrit}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{internal_search_space}}{(\link[paradox:ParamSet]{paradox::ParamSet})\cr
The search space containing those parameters that are internally optimized by the \link[mlr3:Learner]{mlr3::Learner}.}
}
\if{html}{\out{</div>}}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{result_learner_param_vals}}{(\code{list()})\cr
List of param values for the optimal learner call.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-TuningInstanceBatchMultiCrit-new}{\code{TuningInstanceBatchMultiCrit$new()}}
\item \href{#method-TuningInstanceBatchMultiCrit-assign_result}{\code{TuningInstanceBatchMultiCrit$assign_result()}}
\item \href{#method-TuningInstanceBatchMultiCrit-clone}{\code{TuningInstanceBatchMultiCrit$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="clear"><a href='../../bbotk/html/OptimInstance.html#method-OptimInstance-clear'><code>bbotk::OptimInstance$clear()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="format"><a href='../../bbotk/html/OptimInstance.html#method-OptimInstance-format'><code>bbotk::OptimInstance$format()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstance" data-id="print"><a href='../../bbotk/html/OptimInstance.html#method-OptimInstance-print'><code>bbotk::OptimInstance$print()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstanceBatch" data-id="eval_batch"><a href='../../bbotk/html/OptimInstanceBatch.html#method-OptimInstanceBatch-eval_batch'><code>bbotk::OptimInstanceBatch$eval_batch()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="OptimInstanceBatch" data-id="objective_function"><a href='../../bbotk/html/OptimInstanceBatch.html#method-OptimInstanceBatch-objective_function'><code>bbotk::OptimInstanceBatch$objective_function()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TuningInstanceBatchMultiCrit-new"></a>}}
\if{latex}{\out{\hypertarget{method-TuningInstanceBatchMultiCrit-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TuningInstanceBatchMultiCrit$new(
  task,
  learner,
  resampling,
  measures,
  terminator,
  search_space = NULL,
  store_benchmark_result = TRUE,
  store_models = FALSE,
  check_values = FALSE,
  callbacks = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{task}}{(\link[mlr3:Task]{mlr3::Task})\cr
Task to operate on.}

\item{\code{learner}}{(\link[mlr3:Learner]{mlr3::Learner})\cr
Learner to tune.}

\item{\code{resampling}}{(\link[mlr3:Resampling]{mlr3::Resampling})\cr
Resampling that is used to evaluate the performance of the hyperparameter configurations.
Uninstantiated resamplings are instantiated during construction so that all configurations are evaluated on the same data splits.
Already instantiated resamplings are kept unchanged.
Specialized \link{Tuner} change the resampling e.g. to evaluate a hyperparameter configuration on different data splits.
This field, however, always returns the resampling passed in construction.}

\item{\code{measures}}{(list of \link[mlr3:Measure]{mlr3::Measure})\cr
Measures to optimize.}

\item{\code{terminator}}{(\link[bbotk:Terminator]{bbotk::Terminator})\cr
Stop criterion of the tuning process.}

\item{\code{search_space}}{(\link[paradox:ParamSet]{paradox::ParamSet})\cr
Hyperparameter search space. If \code{NULL} (default), the search space is
constructed from the \link[paradox:to_tune]{paradox::TuneToken} of the learner's parameter set
(learner$param_set).}

\item{\code{store_benchmark_result}}{(\code{logical(1)})\cr
If \code{TRUE} (default), store resample result of evaluated hyperparameter
configurations in archive as \link[mlr3:BenchmarkResult]{mlr3::BenchmarkResult}.}

\item{\code{store_models}}{(\code{logical(1)})\cr
If \code{TRUE}, fitted models are stored in the benchmark result
(\code{archive$benchmark_result}). If \code{store_benchmark_result = FALSE}, models
are only stored temporarily and not accessible after the tuning. This
combination is needed for measures that require a model.}

\item{\code{check_values}}{(\code{logical(1)})\cr
If \code{TRUE}, hyperparameter values are checked before evaluation and
performance scores after. If \code{FALSE} (default), values are unchecked but
computational overhead is reduced.}

\item{\code{callbacks}}{(list of \link[mlr3misc:Callback]{mlr3misc::Callback})\cr
List of callbacks.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TuningInstanceBatchMultiCrit-assign_result"></a>}}
\if{latex}{\out{\hypertarget{method-TuningInstanceBatchMultiCrit-assign_result}{}}}
\subsection{Method \code{assign_result()}}{
The \link{Tuner} object writes the best found points and estimated performance values here.
For internal use.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TuningInstanceBatchMultiCrit$assign_result(
  xdt,
  ydt,
  learner_param_vals = NULL,
  xydt = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{xdt}}{(\code{data.table::data.table()})\cr
Hyperparameter values as \code{data.table::data.table()}. Each row is one
configuration. Contains values in the search space. Can contain additional
columns for extra information.}

\item{\code{ydt}}{(\code{data.table::data.table()})\cr
Optimal outcomes, e.g. the Pareto front.}

\item{\code{learner_param_vals}}{(List of named \verb{list()s})\cr
Fixed parameter values of the learner that are neither part of the}

\item{\code{xydt}}{(\code{data.table::data.table()})\cr
Point, outcome, and additional information.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TuningInstanceBatchMultiCrit-clone"></a>}}
\if{latex}{\out{\hypertarget{method-TuningInstanceBatchMultiCrit-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TuningInstanceBatchMultiCrit$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
