% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions.R
\name{multiCoef}
\alias{multiCoef}
\title{Wavelet coefficient estimation from a multichannel signal}
\usage{
multiCoef(Y, G = directBlur(nrow(as.matrix(Y)), ncol(as.matrix(Y))),
  alpha = rep(1, dim(as.matrix(Y))[2]),
  resolution = resolutionMethod(detectBlur(G)), j0 = 3L, j1 = NA_integer_,
  eta = NA_real_, deg = 3L)
}
\arguments{
\item{Y}{An input signal either an n by m matrix containing the multichannel signal to be analysed or single vector of n elements for the single channel. In the multichannel case, each of the m columns represents a channel of n observations.}

\item{G}{The input multichannel blur matrix/vector (needs to be the same dimension/length as the signal input which is a matrix or vector for the multichannel or single channel case respectively). This argument dictates the form of blur present in each of the channels.}

\item{alpha}{A numeric vector, with m elements, specifying the level of long memory for the noise process within each channel of the form alpha = 2 - 2H, where H is the Hurst parameter. If alpha is a single element, that same element is repeated across all required channels.}

\item{resolution}{A character string describing which resolution selection method is to be applied.\itemize{
\item 'smooth': Smooth stopping rule in Fourier domain applied piecewise in each channel and maximum selected which is appropriate if blurring kernel is of regular smooth blur type or direct model (no convolution).
\item 'block': Blockwise variance selection method is used which is appropriate for box car type.}
The default choice uses the detectBlur function to identify what type of blur matrix, G, is input and then maps that identification to the resolution type via a simple switch statement in the hidden \code{resolutionMethod} function, whereby, identified 'smooth' and 'direct' blur use the smooth resolution selection while box.car uses the blockwise resolution selection method.}

\item{j0}{The coarsest resolution level for the wavelet expansion.}

\item{j1}{The finest resolution level for the wavelet expansion. If unspecified, the function will compute all thresholds up to the maximum possible resolution level at j1 = log2(n) - 1.}

\item{eta}{The smoothing parameter. The default level is \eqn{2\sqrt(\alpha^*)} where \eqn{\alpha^*} is an optimal level depending on the type of blur. (see Kulik, Sapatinas and Wishart (2014) for details and justification)}

\item{deg}{The degree of the auxiliary polynomial used in the Meyer wavelet.}
}
\description{
Estimates the wavelet coefficients for the underlying signal of interest embedded in the noisy multichannel deconvolution model.
}
\details{
Returns an object of type \emph{waveletCoef} which is a list including the following three objects \itemize{
\item \code{coef} a numeric vector of size n giving the estimated wavelet coefficients for the signal of interest
\item \code{j0} an integer that specifies the initial coarse resolution for the inhomogeneous wavelet expansion.
\item \code{deg} an integer that specifies the degree of the Meyer wavelet used in the estimation of the wavelet coefficients.
}
}
\examples{
library(mwaved)
# Simulate the multichannel doppler signal.
m <- 3
n <- 2^10
signal <- makeDoppler(n)
# Noise levels per channel
e <- rnorm(m*n)
# Create Gamma blur
shape <- seq(from = 0.5, to = 1, length = m)
scale <- rep(0.25,m)
G <- gammaBlur(n, shape = shape, scale = scale)
# Convolve the signal
X <- blurSignal(signal, G)
# Create error with custom signal to noise ratio
SNR <- c(10,15,20)
sigma <- sigmaSNR(X, SNR)
if (requireNamespace("fracdiff", quietly = TRUE)) {
  alpha <- c(0.75, 0.8, 1)
} else {
  alpha <- rep(1, m)
}
E <- multiNoise(n, sigma, alpha)
# Create noisy & blurred multichannel signal
Y <- X + E
plot(signal, type='l', lty='dashed', main='dashed: True signal, solid: multichannel signals')
matlines(Y, lty = 1)
# Estimate the wavelet coefficients
estimatedCoefs <- multiCoef(Y, G, alpha = alpha)
plot(estimatedCoefs)
# Compute true wavelet coefficients
trueCoefs <- multiCoef(signal)
plot(trueCoefs)
}
