% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/std.R
\name{std}
\alias{std}
\title{Standardizes a design matrix}
\usage{
std(X, Xnew)
}
\arguments{
\item{X}{A matrix (or object that can be coerced to a matrix, such as a data frame or numeric vector).}

\item{Xnew}{Optional. If supplied, \code{X} must be the output of \code{std()} and \code{Xnew} is to be standardized in the same way. See examples for why this might be useful.}
}
\value{
The standardized design matrix, with the following attribues:
\itemize{
\item \code{center}, \code{scale}: mean and standard deviation used to scale the columns
\item \code{nonsingular}: A vector indicating which columns of the original design matrix were able to be standardized (constant columns cannot be standardized to have a standard deviation of 1)
}
}
\description{
The function \code{std} accepts a design matrix and returns a standardized version of that matrix (i.e., each column will have mean 0 and mean sum of squares equal to 1).
}
\details{
This function centers and scales each column of \code{X} so that
\deqn{\sum_{i=1}^n x_{ij}=0}{sum(X[,j])=0}
and
\deqn{n^{-1} \sum_{i=1}^n x_{ij}^2 = 1}{mean(X[,j]^2)=1}
for all j.  This is usually not necessary to call directly, as \code{ncvreg} internally standardizes the design matrix, but inspection of the standardized design matrix can sometimes be useful.  This differs from the base R function \code{\link[base]{scale}} in two ways:
\enumerate{
\item \code{scale} uses the sample standard deviation \code{sqrt(sum(x^2)/(n-1))}, while \code{std} uses the root-mean-square (population) standard deviation \code{sqrt(mean(sum(x^2)))}
\item \code{std} is faster.
}
}
\examples{
data(Prostate)
S <- std(Prostate$X)
apply(S, 2, sum)
apply(S, 2, function(x) mean(x^2))

# Standardizing new observations
X1 <- Prostate$X[1:90,]
X2 <- Prostate$X[91:97,]
S <- std(X1)
head(std(S, X2))
# Useful if you fit to a standardized X, but then get new obs:
y <- Prostate$y[1:90]
fit <- ncvreg(S, y)
predict(fit, std(S, X2), lambda=0.1)
# Same as
predict(ncvreg(X1, y), X2, lambda=0.1)
}
