% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NNCTFunctions.R
\name{nnct}
\alias{nnct}
\title{Nearest Neighbor Contingency Table (NNCT)}
\usage{
nnct(x, lab, ties = FALSE, is.ipd = TRUE, ...)
}
\arguments{
\item{x}{The IPD matrix (if \code{is.ipd=TRUE}) or a data set of points in matrix or data frame form where points
correspond to the rows (if \code{is.ipd = FALSEALSE}).}

\item{lab}{The \code{vector} of class labels (numerical or categorical)}

\item{ties}{A logical argument (default=\code{FALSE}) to take ties into account or not. If \code{TRUE} a NN 
contributes \eqn{1/m} to the NN count if it is one of the \eqn{m} tied NNs of a subject.}

\item{is.ipd}{A logical parameter (default=\code{TRUE}). If \code{TRUE}, \code{x} is taken as the inter-point distance
matrix, otherwise, \code{x} is taken as the data set with rows representing the data points.}

\item{\dots}{are for further arguments, such as \code{method} and \code{p}, passed to the \code{\link[stats]{dist}} function.}
}
\value{
Returns the \eqn{k \times k} NNCT where \eqn{k} is the number of classes in the data set.
}
\description{
Returns the \eqn{k \times k} NNCT given the IPD matrix or data set \code{x} where \eqn{k} is 
the number of classes in the data set.
Rows and columns of the NNCT are labeled with the corresponding class labels.

The argument \code{ties} is a logical argument (default=\code{FALSE}) to take ties into account or not.
If \code{TRUE} a NN 
contributes \eqn{1/m} to the NN count if it is one of the \eqn{m} tied NNs of a subject.

The argument \code{is.ipd} is a logical argument (default=\code{TRUE}) to determine the structure of the argument \code{x}.
If \code{TRUE}, \code{x} is taken to be the inter-point distance (IPD) matrix, and if \code{FALSE}, \code{x} is taken to be the data set
with rows representing the data points.

See also (\insertCite{dixon:1994,dixon:NNCTEco2002,ceyhan:eest-2010,ceyhan:jkss-posthoc-2017;textual}{nnspat})
and the references therein.
}
\examples{
n<-20  #or try sample(1:20,1)
Y<-matrix(runif(3*n),ncol=3)
ipd<-ipd.mat(Y)
cls<-sample(1:2,n,replace = TRUE)  #or try cls<-rep(1:2,c(10,10))
nnct(ipd,cls)
nnct(ipd,cls,ties = TRUE)

nnct(Y,cls,is.ipd = FALSE)
nnct(Y,cls,is.ipd = FALSE,method="max")
nnct(Y,cls,is.ipd = FALSE,method="mink",p=6)

#with one class, it works but really uninformative
cls<-rep(1,n)
nnct(ipd,cls)

#cls as a factor
na<-floor(n/2); nb<-n-na
fcls<-rep(c("a","b"),c(na,nb))
nnct(ipd,fcls)

#cls as an unsorted factor
fcls1<-sample(c("a","b"),n,replace = TRUE)
nnct(ipd,fcls1)

fcls2<-sort(fcls1)
nnct(ipd,fcls2) #ipd needs to be sorted as well, otherwise this result will not agree with fcls1

nnct(Y,fcls1,ties = TRUE,is.ipd = FALSE)

#############
n<-40
Y<-matrix(runif(3*n),ncol=3)
ipd<-ipd.mat(Y)
cls<-sample(1:4,n,replace = TRUE)  #or try cls<-rep(1:2,c(10,10))
nnct(ipd,cls)
nnct(Y,cls,is.ipd = FALSE)

#cls as a factor
fcls<-rep(letters[1:4],rep(10,4))
nnct(ipd,fcls)

#1D data points
n<-20  #or try sample(1:20,1)
X<-as.matrix(runif(n))# need to be entered as a matrix with one column
#(i.e., a column vector), hence X<-runif(n) would not work
ipd<-ipd.mat(X)
cls<-sample(1:2,n,replace = TRUE)  #or try cls<-rep(1:2,c(10,10))
nnct(ipd,cls)

#cls as a factor
na<-floor(n/2); nb<-n-na
fcls<-rep(c("a","b"),c(na,nb))
nnct(ipd,fcls)

#with possible ties in the data
Y<-matrix(round(runif(3*n)*10),ncol=3)
ipd<-ipd.mat(Y)
cls<-sample(1:2,n,replace = TRUE)  #or try cls<-rep(1:2,c(10,10))
nnct(ipd,cls)
nnct(ipd,cls,ties = TRUE)

}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{nnct.sub}}, \code{\link{scct}}, \code{\link{rct}}, and \code{\link{tct}}
}
\author{
Elvan Ceyhan
}
