% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ocf.R
\name{ocf}
\alias{ocf}
\title{Ordered Correlation Forest}
\usage{
ocf(
  Y = NULL,
  X = NULL,
  honesty = FALSE,
  honesty.fraction = 0.5,
  inference = FALSE,
  alpha = 0,
  n.trees = 2000,
  mtry = ceiling(sqrt(ncol(X))),
  min.node.size = 5,
  max.depth = 0,
  replace = FALSE,
  sample.fraction = ifelse(replace, 1, 0.5),
  n.threads = 1
)
}
\arguments{
\item{Y}{Outcome vector.}

\item{X}{Covariate matrix (no intercept).}

\item{honesty}{Whether to grow honest forests.}

\item{honesty.fraction}{Fraction of honest sample. Ignored if \code{honesty = FALSE}.}

\item{inference}{Whether to extract weights and compute standard errors. The weights extraction considerably slows down the routine. \code{honesty = TRUE} is required for valid inference.}

\item{alpha}{Controls the balance of each split. Each split leaves at least a fraction \code{alpha} of observations in the parent node on each side of the split.}

\item{n.trees}{Number of trees.}

\item{mtry}{Number of covariates to possibly split at in each node. Default is the square root of the number of covariates.}

\item{min.node.size}{Minimal node size.}

\item{max.depth}{Maximal tree depth. A value of 0 corresponds to unlimited depth, 1 to "stumps" (one split per tree).}

\item{replace}{If \code{TRUE}, grow trees on bootstrap subsamples. Otherwise, trees are grown on random subsamples drawn without replacement.}

\item{sample.fraction}{Fraction of observations to sample.}

\item{n.threads}{Number of threads. Zero corresponds to the number of CPUs available.}
}
\value{
Object of class \code{ocf}.
}
\description{
Nonparametric estimator for ordered non-numeric outcomes. The estimator modifies a standard random forest
splitting criterion to build a collection of forests, each estimating the conditional probability of a single class.
}
\examples{
\donttest{## Generate synthetic data.
set.seed(1986)

data <- generate_ordered_data(100)
sample <- data$sample
Y <- sample$Y
X <- sample[, -1]

## Training-test split.
train_idx <- sample(seq_len(length(Y)), floor(length(Y) * 0.5))

Y_tr <- Y[train_idx]
X_tr <- X[train_idx, ]

Y_test <- Y[-train_idx]
X_test <- X[-train_idx, ]

## Fit ocf on training sample.
forests <- ocf(Y_tr, X_tr)

## We have compatibility with generic S3-methods.
print(forests)
summary(forests)
predictions <- predict(forests, X_test)
head(predictions$probabilities)
table(Y_test, predictions$classification)

## Compute standard errors. This requires honest forests.
honest_forests <- ocf(Y_tr, X_tr, honesty = TRUE, inference = TRUE)
head(honest_forests$predictions$standard.errors)

## Marginal effects.
me <- marginal_effects(forests, eval = "atmean")
print(me)
print(me, latex = TRUE)

## Compute standard errors. This requires honest forests.
honest_me <- marginal_effects(honest_forests, eval = "atmean", inference = TRUE)
honest_me$standard.errors
print(honest_me, latex = TRUE)}

}
\references{
\itemize{
  \item Di Francesco, R. (2023). Ordered Correlation Forest. arXiv preprint \href{https://arxiv.org/abs/2309.08755}{arXiv:2309.08755}.
}
}
\seealso{
\code{\link{marginal_effects}}
}
\author{
Riccardo Di Francesco
}
