% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimation.R
\name{pdSpecEst}
\alias{pdSpecEst}
\title{Wavelet-thresholded multivariate spectral estimator}
\usage{
pdSpecEst(P, lam = NULL, order = 5, return = "f", alpha = 0.75)
}
\arguments{
\item{P}{a (\eqn{d,d,m})-dimensional array of Hermitian PD matrices, with \eqn{m} a dyadic number.}

\item{lam}{an optional argument specifying the wavelet threshold, if \code{lam}
is not specified the threshold is calculated by a twofold cross-validation procedure.}

\item{order}{an odd integer between 1 and 9 corresponding to the refinement order of the MI wavelet transform.}

\item{return}{an optional argument that specifies whether the denoised spectral estimator
is returned or not.}

\item{alpha}{an optional argument tuning the weights used to normalize the variances
of the wavelet coefficients across scales. By default, \code{alpha} is set to \code{0.75}.}
}
\value{
The function returns a list with four components:
\item{f }{a (\eqn{d,d,m})-dimensional array corresponding to the wavelet-denoised Hermitian PD (\eqn{d \times d})-dimensional
spectral estimate at the \code{m} different frequencies. If \code{!(return == 'f')}, the inverse wavelet transform
of the thresholded wavelet coefficients is not computed and \code{f} is set equal to \code{NULL}.}
\item{D }{a list of arrays, each (\eqn{d, d, 2^j})-dimensional array contains the thresholded
(\eqn{d \times d})-dimensional wavelet coefficients at the \eqn{2^j} different locations in the given wavelet scale
\eqn{j}. The first list element contains the midpoints at the coarsest scale in the
midpoint pyramid \eqn{j=1}, see (Chau and von Sachs, 2017) for more details.}
\item{lam }{the hard threshold used to threshold the components of the wavelet coefficients.}
\item{components }{a list with two elements. The first element \code{thresholded} is a list of arrays; each (\eqn{d^2, 2^j})-dimensional
array contains the components of the thresholded (\eqn{d \times d})-dimensional wavelet coefficients in terms of an
orthonormal basis of the space of (\eqn{d \times d})-dimensional Hermitian matrices. The columns correspond to the
\eqn{d^2} basis components at each of the \eqn{2^j} different locations at wavelet scale \eqn{j}.  Analogous, for the second element
\code{non-thresholded}, but containing the components of the non-thresholded wavelet coefficients.}
}
\description{
\code{pdSpecEst} calculates a \eqn{(d \times d)}-dimensional Hermitian PD wavelet-denoised multivariate
spectral estimator by thresholding wavelet coefficients in the manifold wavelet domain. The
estimation procedure is described in detail (Chau and von Sachs, 2017a).
}
\details{
The input array \code{P} corresponds to an initial noisy Hermitian PD spectral estimate of the
(\eqn{d \times d})-dimensional spectral matrix at \code{m} different frequencies, with \eqn{m = 2^J} for some
\eqn{J > 0}. This can be e.g. the curve of averaged periodograms given as output in \code{\link{pdPgram}}.\cr
\code{P} is transformed to the manifold wavelet domain by the function \code{\link{WavTransf}}, and the
wavelet coefficients are decomposed in terms of an orthonormal basis of the space of Hermitian matrices. \cr
The variances of the components of the wavelet coefficients are standardized across scales via weighted
log-linear regression, where the weights increase exponentially across scales at a rate depending on the
tuning parameter \code{alpha}. Without prior knowledge of the sparsity of the signal, a choice \code{alpha}
in \eqn{[0.5, 1)} is reasonable in most settings. If \code{alpha} is not specified, by default \code{alpha = 0.75} \cr
The components of the wavelet coefficients are thresholded based on the hard (keep-or-kill)
threshold \code{lam}. If \code{lam} is unspecified, the threshold is determined in a data-adaptive manner
by a twofold cross-validation procedure, which is described in detail in (Chau and von Sachs, 2017). \cr
If \code{return == 'f'} the thresholded wavelet coefficients are transformed back to the frequency domain by
the function \code{\link{InvWavTransf}} giving the wavelet-denoised Hermitian PD spectral estimate.
}
\examples{
## ARMA(1,1) process: Example 11.4.1 in (Brockwell and Davis, 1991)

Phi <- array(c(0.7, 0, 0, 0.6, rep(0, 4)), dim = c(2, 2, 2))
Theta <- array(c(0.5, -0.7, 0.6, 0.8, rep(0, 4)), dim = c(2, 2, 2))
Sigma <- matrix(c(1, 0.71, 0.71, 2), nrow = 2)
ts.sim <- rARMA(2^10, 2, Phi, Theta, Sigma)
ts.plot(ts.sim$X) # plot generated time series traces.

pgram <- pdPgram(ts.sim$X)
f <- pdSpecEst(pgram$P)

}
\references{
Chau, J. and von Sachs, R. (2017a). \emph{Positive definite multivariate spectral
estimation: a geometric wavelet approach}. Available at \url{http://arxiv.org/abs/1701.03314}.

Brockwell, P.J. and Davis, R.A. (1991). \emph{Time series: Theory and Methods}. New York: Springer.
}
\seealso{
\code{\link{pdPgram}}, \code{\link{WavTransf}}, \code{\link{InvWavTransf}}
}
