% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pif_plot.R
\name{pif.plot}
\alias{pif.plot}
\title{Plot of Potential Impact Fraction under different values of Relative 
  Risk's parameter theta}
\usage{
pif.plot(X, thetalow, thetaup, rr, cft = NA, method = "empirical",
  confidence_method = "bootstrap", confidence = 95, nsim = 100,
  weights = rep(1/nrow(as.matrix(X)), nrow(as.matrix(X))), mpoints = 100,
  adjust = 1, n = 512, Xvar = var(X), deriv.method.args = list(),
  deriv.method = "Richardson", ktype = "gaussian", bw = "SJ",
  colors = c("deepskyblue", "gray25"), xlab = "Theta", ylab = "PIF",
  title = "Potential Impact Fraction (PIF) under different values of theta",
  check_exposure = TRUE, check_rr = TRUE, check_integrals = TRUE,
  check_cft = TRUE, check_thetas = TRUE, check_xvar = TRUE,
  is_paf = FALSE)
}
\arguments{
\item{X}{Random sample (\code{data.frame}) which includes exposure 
and covariates or sample \code{mean} if \code{"approximate"} method is 
selected.}

\item{thetalow}{Minimum of \code{theta} (parameter of relative risk 
\code{rr}) for plot.}

\item{thetaup}{Maximum of \code{theta} (parameter of relative risk 
\code{rr}) for plot.}

\item{rr}{\code{function} for Relative Risk which uses parameter 
 \code{theta}. The order of the parameters should be \code{rr(X, theta)}.
 
\strong{**Optional**}}

\item{cft}{Function \code{cft(X)} for counterfactual. Leave empty for 
the Population Attributable Fraction \code{\link{paf}} where 
counterfactual is that of the theoretical minimum risk exposure 
\code{X0} such that \code{rr(X0,theta) = 1}.}

\item{method}{Either \code{"empirical"} (default), \code{"kernel"} or 
\code{"approximate"}. For details on estimation methods see 
\code{\link{pif}}.}

\item{confidence_method}{Either \code{bootstrap} (default), \code{linear}, 
\code{loglinear}. See \code{\link{paf}} details for additional information.}

\item{confidence}{Confidence level \% (default \code{95}).}

\item{nsim}{Number of simulations to generate confidence intervals.}

\item{weights}{Normalized survey \code{weights} for the sample \code{X}.}

\item{mpoints}{Number of points in plot.}

\item{adjust}{Adjust bandwith parameter (for \code{"kernel"} 
method) from \code{\link[stats]{density}}.}

\item{n}{Number of equally spaced points at which the density (for 
\code{"kernel"} method) is to be estimated (see 
\code{\link[stats]{density}}).}

\item{Xvar}{Variance of exposure levels (for \code{"approximate"} 
method).}

\item{deriv.method.args}{\code{method.args} for 
\code{\link[numDeriv]{hessian}} (for \code{"approximate"} method).}

\item{deriv.method}{\code{method} for \code{\link[numDeriv]{hessian}}. 
Don't change this unless you know what you are doing (for 
\code{"approximate"} method).}

\item{ktype}{\code{kernel} type:  \code{"gaussian"}, 
\code{"epanechnikov"}, \code{"rectangular"}, \code{"triangular"}, 
\code{"biweight"}, \code{"cosine"}, \code{"optcosine"} (for \code{"kernel"}
method). Additional information on kernels in \code{\link[stats]{density}}.}

\item{bw}{Smoothing bandwith parameter (for 
\code{"kernel"} method) from \code{\link[stats]{density}}. Default 
\code{"SJ"}.}

\item{colors}{\code{vector} Colours of plot.}

\item{xlab}{\code{string} Label of x-axis in plot.}

\item{ylab}{\code{string} Label of y-axis in plot.}

\item{title}{\code{string} Title of plot.}

\item{check_exposure}{\code{boolean}  Check that exposure \code{X} is 
positive and numeric.}

\item{check_rr}{\code{boolean} Check that Relative Risk function \code{rr} equals 
\code{1} when evaluated at \code{0}.}

\item{check_integrals}{\code{boolean}  Check that counterfactual \code{cft} 
and relative risk's \code{rr} expected values are well defined for this 
scenario.}

\item{check_cft}{\code{boolean}  Check that counterfactual function 
\code{cft} reduces exposure.}

\item{check_thetas}{\code{boolean} Check that theta associated parameters are
correctly inputed for the model.}

\item{check_xvar}{\code{boolean} Check \code{Xvar} is covariance matrix.}

\item{is_paf}{Boolean forcing evaluation of \code{\link{paf}}. This forces
the \code{pif} function ignore the inputed counterfactual and set it to the
theoretical minimum risk value of \code{1}.}
}
\value{
pif.plot       \code{\link[ggplot2]{ggplot}} object with plot of 
  Potential Impact Fraction as function of \code{theta}.
}
\description{
Function that plots the \code{\link{pif}} under different values
  of a univariate parameter \code{theta} of the relative risk function \code{rr} 
  which depends on the exposure \code{X} and a  parameter \code{theta}
  (\code{rr(X, theta)})
}
\examples{

#Example 1: Exponential Relative Risk empirical method
#-----------------------------------------------------
\dontrun{
set.seed(18427)
X <- data.frame(Exposure = rbeta(25, 4.2, 10))
pif.plot(X, thetalow = 0, thetaup = 10, rr =  function(X, theta){exp(theta*X)})

#Same example with kernel method
pif.plot(X, thetalow = 0, thetaup = 10, rr =  function(X, theta){exp(theta*X)}, 
method = "kernel", title = "Kernel method example") 
 
#Same example for approximate method. Notice that approximate method has 
#more uncertainty
Xmean <- data.frame(mean(X[,"Exposure"]))
Xvar  <- var(X)
pif.plot(Xmean, thetalow = 0, thetaup = 10, rr =  function(X, theta){exp(theta*X)}, 
method = "approximate", Xvar = Xvar, title = "Approximate method example")

#Example with counterfactual
pif.plot(X, thetalow = -10, thetaup = -5, rr = function(X, theta){exp(theta*X)}, 
cft = function(X){sqrt(X)})

#Example for approximate method with square root counterfactual
#Notice how the approximate represents information loss and thus the interval
#loses precision.
pif.plot(Xmean, thetalow = -10, thetaup = -5, rr = function(X, theta){exp(theta*X)},  
cft = function(X){sqrt(X)}, method = "approximate", Xvar = Xvar) 
}

}
\seealso{
See \code{\link{pif}} for Potential Impact Fraction estimation with
  confidence intervals \code{\link{pif.confidence}}.
  
See \code{\link{paf.plot}} for same plot with 
Population Attributable Fraction \code{\link{paf}}.
}
\author{
Rodrigo Zepeda-Tello \email{rzepeda17@gmail.com}

Dalia Camacho-García-Formentí \email{daliaf172@gmail.com}
}
