#' @name sprinkle_na_string
#' @title Sprinkle Appearance of NA's
#' 
#' @description The appearance of \code{NA} values in a table may be dependent
#'   on the context.  \code{pixiedust} uses the \code{na_string} sprinkle
#'   to guide the appearance of missing values in the table.
#'   
#' @param x An object of class \code{dust}
#' @param rows Either a numeric vector of rows in the tabular object to be 
#'   modified or an object of class \code{call}.  When a \code{call}, 
#'   generated by \code{quote(expression)}, the expression resolves to 
#'   a logical vector the same length as the number of rows in the table.
#'   Sprinkles are applied to where the expression resolves to \code{TRUE}.
#' @param cols Either a numeric vector of columns in the tabular object to
#'   be modified, or a character vector of column names. A mixture of 
#'   character and numeric indices is permissible.
#' @param na_string \code{character(1)} A character string giving desired
#'   replacement for \code{NA} values in the selected cells.
#' @param part A character string denoting which part of the table to modify.
#' @param fixed \code{logical(1)} indicating if the values in \code{rows} 
#'   and \code{cols} should be read as fixed coordinate pairs.  By default, 
#'   sprinkles are applied at the intersection of \code{rows} and \code{cols}, 
#'   meaning that the arguments do not have to share the same length.  
#'   When \code{fixed = TRUE}, they must share the same length.
#' @param recycle A \code{character} one that determines how sprinkles are 
#'   managed when the sprinkle input doesn't match the length of the region
#'   to be sprinkled.  By default, recycling is turned off.  Recycling 
#'   may be performed across rows first (left to right, top to bottom), 
#'   or down columns first (top to bottom, left to right).
#' @param ... Additional arguments to pass to other methods. Currently ignored.
#' 
#' @section Functional Requirements:
#' \enumerate{
#'  \item Correctly reassigns the appropriate elements \code{na_string} column
#'    in the table part.
#'  \item Casts an error if \code{x} is not a \code{dust} object.
#'  \item Casts an error if \code{bg} is not a \code{character(1)}
#'  \item Casts an error if \code{part} is not one of \code{"body"}, 
#'    \code{"head"}, \code{"foot"}, or \code{"interfoot"}
#'  \item Casts an error if \code{fixed} is not a \code{logical(1)}
#'  \item Casts an error if \code{recycle} is not one of \code{"none"},
#'    \code{"rows"}, or \code{"cols"}
#' }
#' 
#' The functional behavior of the \code{fixed} and \code{recycle} arguments 
#' is not tested for this function. It is tested and validated in the
#' tests for \code{\link{index_to_sprinkle}}.
#' 
#' @seealso \code{\link{sprinkle}}, 
#'   \code{\link{index_to_sprinkle}}
#'
#' @export

sprinkle_na_string <- function(x, rows = NULL, cols = NULL,
                               na_string = getOption("pixie_na_string", NA), 
                               part = c("body", "head", "foot", "interfoot"),
                               fixed = FALSE, 
                               recycle = c("none", "rows", "cols", "columns"),
                               ...)
{
  UseMethod("sprinkle_na_string")
}

#' @rdname sprinkle_na_string
#' @export

sprinkle_na_string.default <- function(x, rows = NULL, cols = NULL,
                                       na_string = getOption("pixie_na_string", NA), 
                                       part = c("body", "head", "foot", "interfoot"),
                                       fixed = FALSE, 
                                       recycle = c("none", "rows", "cols", "columns"),
                                       ...)
{
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_character(x = na_string,
                              len = 1,
                              add = coll)
  
  indices <- index_to_sprinkle(x = x, 
                               rows = rows, 
                               cols = cols, 
                               fixed = fixed,
                               part = part,
                               recycle = recycle,
                               coll = coll)
  
  checkmate::reportAssertions(coll)
  
  # At this point, part should have passed the assertions in 
  # index_to_sprinkle. The first element is expected to be valid.
  
  part <- part[1]
  
  x[[part]][["na_string"]][indices] <- na_string
  
  x
}

#' @rdname sprinkle_na_string
#' @export

sprinkle_na_string.dust_list <- function(x, rows = NULL, cols = NULL,
                                         na_string = getOption("pixie_na_string", NA), 
                                         part = c("body", "head", "foot", "interfoot"),
                                         fixed = FALSE, 
                                         recycle = c("none", "rows", "cols", "columns"),
                                         ...)
{
  structure(
    lapply(X = x,
           FUN = sprinkle_na_string.default,
           rows = rows,
           cols = cols,
           na_string = na_string,
           part = part,
           fixed = fixed,
           recycle = recycle,
           ...),
    class = "dust_list"
  )
}

# Unexported Utility ------------------------------------------------

# These functions are to be used inside of the general `sprinkle` call
# When used inside `sprinkle`, the indices are already determined, 
# the only the `na_string` argument needs to be validated. 
# The assert function is kept separate so it may be called earlier
# without attempting to perform the assignment.

sprinkle_na_string_index_assert <- function(na_string, coll)
{
  checkmate::assert_character(x = na_string,
                              len = 1,
                              add = coll,
                              .var.name = "na_string")
}

sprinkle_na_string_index <- function(x, indices, na_string, part)
{
  x[[part]][["na_string"]][indices] <- na_string
  
  x
}