#' @name sprinkle_pad
#' @title Sprinkle the Padding of a Cell
#' 
#' @description Padding for HTML tables indicates how many pixels should
#' be placed between the cell's content and the outside border.
#'   
#' @param x An object of class \code{dust}
#' @param rows Either a numeric vector of rows in the tabular object to be 
#'   modified or an object of class \code{call}.  When a \code{call}, 
#'   generated by \code{quote(expression)}, the expression resolves to 
#'   a logical vector the same length as the number of rows in the table.
#'   Sprinkles are applied to where the expression resolves to \code{TRUE}.
#' @param cols Either a numeric vector of columns in the tabular object to
#'   be modified, or a character vector of column names. A mixture of 
#'   character and numeric indices is permissible.
#' @param pad \code{numeric(1)} A character string giving a color for the 
#'   background of the chosen cells.  
#' @param part A character string denoting which part of the table to modify.
#' @param fixed \code{logical(1)} indicating if the values in \code{rows} 
#'   and \code{cols} should be read as fixed coordinate pairs.  By default, 
#'   sprinkles are applied at the intersection of \code{rows} and \code{cols}, 
#'   meaning that the arguments do not have to share the same length.  
#'   When \code{fixed = TRUE}, they must share the same length.
#' @param recycle A \code{character} one that determines how sprinkles are 
#'   managed when the sprinkle input doesn't match the length of the region
#'   to be sprinkled.  By default, recycling is turned off.  Recycling 
#'   may be performed across rows first (left to right, top to bottom), 
#'   or down columns first (top to bottom, left to right).
#' @param ... Additional arguments to pass to other methods. Currently ignored.
#'   
#' @details Colors may be a dvips color name, or in the rgb(R, G, B), 
#' rgba(R, G, B, A), #RRGGBB, or #RRGGBBAA formats. See 
#' \url{http://nutterb.github.io/pixiedust/colors.html} for additional
#' details.
#'   
#' This sprinkle is ignored in console and markdown outputs.  HTML output
#' will accept any of the color formats and recognize transparency.  LaTeX
#' output will accept any of the color formats but ignore transparency.
#' 
#' As long as \code{pad} is required to be a \code{numeric(1)}, the 
#' \code{recycle} argument is kind of useless. It is included to maintain
#' consistency with the \code{index_to_sprinkle} function. Future development
#' may permit a character vector of colors.
#' 
#' @section Functional Requirements:
#' \enumerate{
#'  \item Correctly reassigns the appropriate elements \code{pad} column
#'    in the table part.
#'  \item Casts an error if \code{x} is not a \code{dust} object.
#'  \item Casts an error if \code{pad} is not a \code{numeric(1)}
#'  \item Casts an error if \code{part} is not one of \code{"body"}, 
#'    \code{"head"}, \code{"foot"}, or \code{"interfoot"}
#'  \item Casts an error if \code{fixed} is not a \code{logical(1)}
#'  \item Casts an error if \code{recycle} is not one of \code{"none"},
#'    \code{"rows"}, or \code{"cols"}
#' }
#' 
#' The functional behavior of the \code{fixed} and \code{recycle} arguments 
#' is not tested for this function. It is tested and validated in the
#' tests for \code{\link{index_to_sprinkle}}.
#' 
#' @seealso \code{\link{sprinkle}},
#'   \code{\link{index_to_sprinkle}}
#'   
#' @author Benjamin Nutter
#' 
#' @export

sprinkle_pad <- function(x, rows = NULL, cols = NULL, pad, 
                        part = c("body", "head", "foot", "interfoot"),
                        fixed = FALSE, 
                        recycle = c("none", "rows", "cols", "columns"), 
                        ...)
{
  UseMethod("sprinkle_pad")
}

#' @rdname sprinkle_pad
#' @export

sprinkle_pad.default <- function(x, rows = NULL, cols = NULL, pad, 
                                part = c("body", "head", "foot", "interfoot"),
                                fixed = FALSE, 
                                recycle = c("none", "rows", "cols", "columns"), 
                                ...)
{
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_numeric(x = pad,
                            len = 1,
                            add = coll)
  
  indices <- index_to_sprinkle(x = x, 
                               rows = rows, 
                               cols = cols, 
                               fixed = fixed,
                               part = part,
                               recycle = recycle,
                               coll = coll)
  
  checkmate::reportAssertions(coll)
  
  # At this point, part should have passed the assertions in 
  # index_to_sprinkle. The first element is expected to be valid.
  
  part <- part[1]
  
  x[[part]][["pad"]][indices] <- pad
  
  x
  
}

#' @rdname sprinkle_pad
#' @export

sprinkle_pad.dust_list <- function(x, rows = NULL, cols = NULL, pad, 
                                  part = c("body", "head", "foot", "interfoot"),
                                  fixed = FALSE, 
                                  recycle = c("none", "rows", "cols", "columns"), 
                                  ...)
{
  structure(
    lapply(X = x,
           FUN = sprinkle_pad.default,
           rows = rows,
           cols = cols,
           pad = pad,
           part = part,
           fixed = fixed,
           recycle = recycle,
           ...),
    class = "dust_list"
  )
}

# Unexported Utility ------------------------------------------------

# These functions are to be used inside of the general `sprinkle` call
# When used inside `sprinkle`, the indices are already determined, 
# the only the `pad` argument needs to be validated. 
# The assert function is kept separate so it may be called earlier
# without attempting to perform the assignment.

sprinkle_pad_index_assert <- function(pad, coll)
{
  checkmate::assert_numeric(x = pad,
                            len = 1,
                            add = coll,
                            .var.name = "pad")
}

sprinkle_pad_index <- function(x, indices, pad, part)
{
  x[[part]][["pad"]][indices] <- pad
  
  x
}