% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/round_robin.R
\name{rraf}
\alias{rraf}
\title{Round Robin Allele Frequencies}
\usage{
rraf(gid, pop = NULL, res = "list", by_pop = FALSE, correction = TRUE,
  ...)
}
\arguments{
\item{gid}{a genind or genclone object}

\item{pop}{either a formula to set the population factor from the 
\code{\link{strata}} slot or a vector specifying the population factor for 
each sample. Defaults to \code{NULL}.}

\item{res}{either "list" (default), "vector", or "data.frame".}

\item{by_pop}{When this is \code{TRUE}, the calculation will be done by 
population. Defaults to \code{FALSE}}

\item{correction}{a logical indicating whether or not zero-valued allele 
frequencies should be corrected using the methods outlined in 
\link[=rare_allele_correction]{correcting rare alleles}. 
(Default: \code{TRUE})}

\item{...}{options from \link[=rare_allele_correction]{correcting rare
alleles}. The default is to correct allele frequencies to 1/n}
}
\value{
a vector or list of allele frequencies
}
\description{
This function utilizes \code{\link{rrmlg}} to calculate multilocus genotypes 
and then subsets each locus by the resulting MLGs to calculate the 
round-robin allele frequencies used for pgen and psex.
}
\details{
Calculating allele frequencies for clonal populations is a difficult
  task. Frequencies calculated on non-clone-corrected data suffer from bias 
  due to non-independent samples. On the other hand, frequencies calculated 
  on clone-corrected data artificially increases the significance of rare 
  alleles. The method of round-robin allele frequencies as presented in Parks
  and Werth (1993) provides a method of calculating allele frequencies in a 
  way that minimizes both of these effects. 
  \subsection{Rare Alleles}{Allele frequencies at a given locus are
  calculated based on samples that are \strong{clone corrected without that
  locus}. When this happens, rare alleles have a high likelihood of dropping
  out, giving them a frequency of "0". For some analyses, this is a perfectly
  fine outcome, but for analyses such as \code{\link{pgen}} and
  \code{\link{psex}}, this could result in undefined values. Setting 
  \code{correction = TRUE} will allow you to control how these zero-valued 
  allele frequencies are corrected. For details, please see the documentation
  on \link[=rare_allele_correction]{correcting rare alleles} and examples.}
}
\note{
When \code{by_pop = TRUE}, the output will be a matrix of allele 
  frequencies. Additionally, when the argument \code{pop} is not \code{NULL},
  \code{by_pop} is automatically \code{TRUE}.
}
\examples{

data(Pram)

# Round robin allele frequencies, correcting zero-valued frequencies to 1/nInd(Pram)
rraf(Pram)


\dontrun{

## Round robin allele frequencies will be different than observed

# Compare to without round robin:
PrLoc <- seploc(Pram, res = "mat") # get locus by matrix
lapply(PrLoc, colMeans, na.rm = TRUE)

# Without round robin, clone corrected:
Pcc    <- clonecorrect(Pram, strata = NA) # indiscriminantly clone correct
PccLoc <- seploc(Pcc, res = "mat")
lapply(PccLoc, colMeans, na.rm = TRUE)

## Different methods of obtaining round robin allele frequencies

# Get vector output.
rraf(Pram, res = "vector")

# Getting the output as a data frame allows us to use ggplot2 to visualize
(Prdf <- rraf(Pram, res = "data.frame"))
library("ggplot2")
ggplot(Prdf, aes(y = allele, x = frequency)) +
  geom_point() +
  facet_grid(locus ~ ., scale = "free_y", space = "free")

## Round Robin allele frequencies by population (matrix only)

# By default, allele frequencies will be corrected by 1/n per population
(Prbp <- rraf(Pram, by_pop = TRUE))

# This might be problematic because populations like PistolRSF_OR has a 
# population size of four.

# By using the 'e' argument to rare_allele_correction, this can be set to a
# more reasonable value.
(Prbp <- rraf(Pram, by_pop = TRUE, e = 1/nInd(Pram)))



}
}
\references{
Arnaud-Haond, S., Duarte, C. M., Alberto, F., & Serrão, E. A. 2007.
Standardizing methods to address clonality in population studies.
\emph{Molecular Ecology}, 16(24), 5115-5139.

Parks, J. C., & Werth, C. R. 1993. A study of spatial features of clones in a
population of bracken fern, \emph{Pteridium aquilinum} (Dennstaedtiaceae).
\emph{American Journal of Botany}, 537-544.
}
\seealso{
\code{\link{rrmlg}}, \code{\link{pgen}}, \code{\link{psex}},
 \code{\link{rare_allele_correction}}
}
\author{
Zhian N. Kamvar, Jonah Brooks, Stacy A. Krueger-Hadfield, Erik Sotka
}
