### initial settings of two parameters size and mu in a negative binomial
### distribution for a numeric optimal searching function optim in R
SIZE.INIT = 1
MU.INIT = 0.5

### termination conditions for EM algorithm
TOLERANCE = 1e-10
ITER.TOLERANCE = 1e5


### density function of a truncated zero negative binomial distribution
### size and mu are two parameters for the negative binomial
zerotruncated.dnbinom <- function(x, size, mu, log = FALSE)
{
  ## the density of x in negative binomial
  p <- dnbinom(x, size = size, mu = mu, log = log)

  ## set zeros in x with zero probability
  if (log == FALSE) {
    p[ which(x == 0) ] <- 0
  } else {
    p[ which(x == 0) ] <- -Inf
  }

  ## the density of non-zero in negative binomial
  q <- 1 - dnbinom(0, size = size, mu = mu)

  ## normalize all non-zero values in negrative binomial to generate ZTNB
  if (log == FALSE) {
    return( p/q )
  } else {
    return( p - log(q) )
  }
}


### zerotruncated negative loglikelyhood 
zerotruncated.minus.log.likelyhood <- function(hist.table, size, mu)
{
  prob <- zerotruncated.dnbinom(hist.table[, 1], size, mu, log = TRUE)

  ## negative loglikelyhood
  prob <- -prob
  return( prob %*% hist.table[, 2] )
}


### calculate the negative binomial loglikelyhood
### zero.items is number of items unobserved
### size and mu are parameters in a negative binomial distribution
nb.loglikelyhood <- function(hist.table, zero.items, size, mu)
{
  ## likelyhood of nonzero terms
  log.prob <- dnbinom(hist.table[, 1], size = size, mu = mu, log = TRUE)
  loglikelyhood <- log.prob %*% hist.table[, 2]

  ## add items with zero count
  log.zero.prob <- dnbinom(0, size = size, mu = mu, log = TRUE)
  loglikelyhood <- loglikelyhood + zero.items * log.zero.prob

  return(loglikelyhood)
}


### EM algorithm to fit the histogram with a negative binomial distribution
### hist only includes information for observation
### the number of unobserved items is missing data
preseqR.ztnb.em <- function(hist, size=SIZE.INIT, mu=MU.INIT, header=FALSE)
{
  hist.count <- read.hist(hist, header)

  ## setting the number of unobserved items as 0
  zero.prob <- exp(dnbinom(0, size = size, mu = mu, log = TRUE))

  ## estimate the total number of distinct items
  observed.items <- sum(hist.count)
  L <- observed.items/( 1 - zero.prob )

  ## expected the number of unobservations
  zero.items <- L*zero.prob

  ## convert zero.items into an integer
  zero.items <- floor(zero.items)

  ## convert the count vector of a histogram into a histogram table
  freq <- which(hist.count != 0)
  number.items <- hist.count[freq]

  ## estimated mean and variance
  m <- (freq %*% number.items) / L
  v <- ( (freq - m)^2 %*% number.items + m^2 * zero.items )/(L - 1)

  ## build a histogram table
  hist.table <- matrix(c(freq, number.items), ncol = 2, byrow = FALSE)

  ## target function f
  f <- function(x) {
        return( -nb.loglikelyhood(hist.table, zero.items, size = x, mu = m)/L )
  }

  ## derivative of f
  gr <- function(x) 
  {
    first.term <- ( digamma(x) * zero.items + 
                    digamma(hist.table[, 1] + size) %*% hist.table[, 2] )/L
    second.term <- digamma(x)
    third.term <- log(x) - log(x + m)
    result <- first.term - second.term + third.term
    # f is negative loglikelyhood
    return(-result)
  }

  ## estimate size and mu based on first and second moments
  if (v > m) {
    res <- optim(m^2 / (v - m), f, gr, method = "L-BFGS-B",
           lower = 0.0001, upper = 10000)
  } else {
    res <- optim(size, f, gr, method = "L-BFGS-B",
           lower = 0.0001, upper = 10000)
  }

  ## count the times of iteration
  iter <- as.double(1)

  ## initialize the negative loglikelyhood
  loglikelyhood.pre <- Inf

  ## zerotruncated loglikelyhood 
  loglikelyhood <- zerotruncated.minus.log.likelyhood(hist.table, res$par, m)

  ## EM algorithm
  while (( loglikelyhood.pre - loglikelyhood )/observed.items > TOLERANCE &&
           iter < ITER.TOLERANCE)
  {
    ## update negative loglikelyhood
    loglikelyhood.pre <- loglikelyhood

    ## update parameters
    size <- res$par
    mu <- m

### E-step: estimate the number of unobserved items

    ## update the probility an item unobserved
    zero.prob <- exp(dnbinom(0, size = size, mu = mu, log = TRUE))

    ## estimate the total number of distinct items
    L <- observed.items/( 1 - zero.prob )

    ## update expected number of unobserved items
    zero.items <- L*zero.prob

    ## convert zero.items into an integer
    zero.items <- floor(zero.items)

    ## estimated mean and variance
    m <- (freq %*% number.items)/L
    v <- ( (freq - m)^2 %*% number.items + m^2 * zero.items )/(L - 1)

### M step: estimate the parameters size and mu
    if (v > m) {
      res <- optim(m^2 / (v - m), f, gr, method = "L-BFGS-B",
             lower = 0.0001, upper = 10000)
    } else {
      res <- optim(size, f, gr, method = "L-BFGS-B",
             lower = 0.0001, upper = 10000)
    }
    iter <- iter + 1
    ## zerotruncated loglikelyhood
    loglikelyhood <- zerotruncated.minus.log.likelyhood(hist.table, res$par, m)
  }
  return(list(size = size, mu = mu, loglik = -loglikelyhood.pre))
}


### predict the number of distinct items using EM algorithm
### if the histogram file has a header, set header = TRUE
### t is the relative size to inital sample
preseqR.ztnb.estimate <- function(hist, t, header = FALSE)
{
  hist.count <- read.hist(hist, header)

  total.sample <- (1:length(hist.count) %*% hist.count)
  distinct.sample <- sum(hist.count)

  ## convert the count vector of a histogram into a histogram table
  freq = which(hist.count != 0)
  number.items = hist.count[freq]
  hist.table = matrix(c(freq, number.items), ncol = 2, byrow = FALSE)

  ## estimate the parameters
  opt <- preseqR.ztnb.em(hist.table)
  size <- opt$size
  mu <- opt$mu

  ## the probability of being sampled in the initial experiment
  p <- 1 - dnbinom(0, size = size, mu = mu)

  ## L is the estimated total number of distinct items
  L <- distinct.sample/p

  ## update parameters of negative binomial in the experiment with size n
  mu <- mu*(t + 1)

  ## the probability of being sampled under the new experiment
  P <- 1 - dnbinom(0, size = size, mu = mu)

  ## return the expected number of new distinct species under the new experiment
  return(L * P - distinct.sample)
}

## predict a complexity curve using EM algorithm
## ss is the step.size
## max.extrapoltion is the maximum value for extrapolation
preseqR.ztnb.species.accum.curve <- function(hist, ss = NULL,
                                             max.extrapolation = NULL,
                                             header = FALSE)
{
  hist.count <- read.hist(hist, header)

  total.sample <- (1:length(hist.count) %*% hist.count)
  distinct.sample <- sum(hist.count)

  ## set step.size = total.sample if it is undefined
  if (is.null(ss))
    ss <- total.sample

  ## set max.extrapolation = 100 * ss if it is undefined
  if (is.null(max.extrapolation)) {

    ## n is the number of experiments; 100 is a magic number
    max.extrapolation <- 100*total.sample
    n <- as.integer( max.extrapolation/ss )

  } else {

    # n is the number of experiments
    n <- as.integer( max.extrapolation/ss )

  }

  sample.size <- as.double(ss) * (1: n)
  dim(sample.size) <- n

  ## convert the count vector of a histogram into a histogram table
  freq <- which(hist.count != 0)
  number.items <- hist.count[freq]
  hist.table <- matrix(c(freq, number.items), ncol = 2, byrow = FALSE)

  ## estimate parameters
  opt <- preseqR.ztnb.em(hist.table)
  size <- opt$size
  mu <- opt$mu

  ## the probability of being sampled in the initial experiment
  p <- 1 - dnbinom(0, size = size, mu = mu)

  ## L is the estimated total number of distinct items
  L <- distinct.sample/p

  ## estimate the item being sampled under new experiments with different size
  t = sample.size/as.double(total.sample)
  dim(t) = length(t)
  P = apply(t, 1, function(x) 1 - dnbinom(0, size, mu = x * mu))
  yield.estimates = L*P

  ## combine sample.size and yield.estimates into matrix
  yield.estimates = matrix(c(sample.size, yield.estimates), ncol = 2, byrow = FALSE)
  colnames(yield.estimates) = c('sample.size', 'yield.estimate')
  return(yield.estimates)
}
