# Framework for the ProbeMapper package. 
# 
# Author: Jeff Allen (Jeffrey.Allen@UTSouthwestern.edu)
###############################################################################

#' Sets up a variable for the connection to the API. You should pass such a connection variable to all future calls.
#' @param url The base URL at which the API is running, i.e. "http://qbri.swmed.edu/etp/" 
#' @param key the API key which you will use to connect to the API.
#' @returnType list
#' @return a connection object representing the connection to the LCDB API based on the input you've provided.
#' @author Jeffrey D. Allen \email{Jeffrey.Allen@@UTSouthwestern.edu}
#' @export
pm.connect <- function (url, key="none"){
	if (substr(url, nchar(url), nchar(url)) != "/"){
		url <- paste(url, "/", sep="");
	}
	
	DATABASE_API_URL <- url;
	
	API_KEY <- key;
	API_KEY_STR <- paste("key=",API_KEY,sep="");
	
	
	return(list(url=url,keyStr=API_KEY_STR))
}

#' Gets a list of the available platforms in the database
#' @param conn the connection object generated by pm.connect()
#' @returnType list
#' @return a list of the available platforms with their IDs as the row.names
#' @author Jeffrey D. Allen \email{Jeffrey.Allen@@UTSouthwestern.edu}
#' @export
pm.getPlatforms <- function (conn){
	url <- paste(conn["url"], "getPlatforms?",conn["keyStr"], sep="");
	
	dat <- fromJSON(file=url);
	dat <- data.frame(t(sapply(dat, "[")))
	
	return(dat);
}


#' Gets information about a particular gene
#' @param conn the connection object generated by pm.connect()
#' @param entrezID the EntrezID of the gene in which you're interested 
#' @returnType list
#' @return information about the gene including its name, description, symbols, and accession numbers.
#' @author Jeffrey D. Allen \email{Jeffrey.Allen@@UTSouthwestern.edu}
#' @export
pm.getGene <- function(conn, entrezID){
		
	url <- paste(conn["url"], "getGene?",conn["keyStr"], sep="");
	
	if (missing(entrezID)){
		stop("You must provide an EntrezID");
	}
	
	url <- paste(url, "&eid=", entrezID, sep="");
	
	dat <- fromJSON(file=url);
	
	return(dat);
}

#' Gets information about a particular probe
#' @param conn the connection object generated by pm.connect()
#' @param probeID the internal probeID used to uniquely reference probes
#' @param platformID the ID of the platform on which this probeName is found. 
#' @param probeName the vendor-supplied name of the probe for this platformID.
#' @returnType list
#' @return information about the probe specified including its probe ID, platformID, and probe name.
#' @author Jeffrey D. Allen \email{Jeffrey.Allen@@UTSouthwestern.edu}
#' @export
pm.getProbe <- function(conn, probeID, platformID, probeName){
	url <- paste(conn["url"], "getProbe?",conn["keyStr"], sep="");
	
	if (missing(probeID)){
		if (missing(platformID) || missing(probeName)){
			stop("You must provide either a probeID or a probeName + platformID combination.");
		}
		else{
			if (typeof(probeName)=="character"){
				probeName <- URLencode(probeName);
			}
			url <- paste(url, "&plid=",platformID,"&prname=",probeName,sep="");
		}
	}
	else{
		url <- paste(url, "&prid=",probeID,sep="");
		
		if (!missing(platformID) || !missing(probeName)){
			warning("The probeID was provided, so probeName and platformID will be ignored.");
		}
	}
	
	dat <- fromJSON(file=url);
	
	return(dat);
	
}


#' Gets a list of the genes associated with the specified probe
#' @param conn the connection object generated by pm.connect() 
#' @param probeID the internal probeID used to uniquely reference probes
#' @param platformID the ID of the platform associated with the probe name. 
#' @param probeName the name of the probe, as assigned by the vendor, for this platformID.
#' @param authorityID a number of vector of numbers representing the ID(s) of the authorities (sources) 
#' which you want to use to map this probe to genes. At the time of writing, 1 = BLAST, 2 = Vendor, 3 = Bioconductor.
#' @returnType list
#' @return a list containing information about the probe specified and a data.frame detailing the genes with which 
#' this probe is associated. There is a normalized [0-1] value representing the strength of that association according 
#' to each authority in the database.
#' @author Jeffrey D. Allen \email{Jeffrey.Allen@@UTSouthwestern.edu}
#' @export
pm.getGenesByProbe <- function (conn, probeID, platformID, probeName, authorityID){	
		
	url <- paste(conn["url"], "getGenes?",conn["keyStr"], sep="");
	
	if (missing(probeID)){
		if (missing(platformID) || missing(probeName)){
			stop("You must provide either a probeID or a probeName + platformID combination.");
		}
		else{
			if (typeof(probeName)=="character"){
				probeName <- URLencode(probeName);
			}
			url <- paste(url, "&plid=",platformID,"&prname=",probeName,sep="");
		}
	}
	else{
		url <- paste(url, "&prid=",probeID,sep="");
		
		if (!missing(platformID) || !missing(probeName)){
			warning("The probeID was provided, so probeName and platformID will be ignored.");
		}
	}
	
	if (!missing(authorityID)){
		authorityID <- paste(authorityID, collapse=",");
		url <- paste(url, "&authid=", authorityID, sep="");
	}
	
	nullToNA <- function(x) sapply(x, function(x) if (is.list(x)) nullToNA(x) else if (is.null(x)) NA else x,simplify=FALSE)
			
	dat <- fromJSON(file=url);
	
	dat <- nullToNA(dat);
	
	dat$genes <- data.frame(t(sapply(dat$genes, "[")))
	
	return(dat);
}



#' Get the probes associated with a specific gene
#' @param conn the connection object generated by pm.connect()
#' @param entrezID the EntrezID of the gene of interest.
#' @param platformID a number or vector of numbers representing the ID(s) of the platforms from which you want to find 
#' probes.
#' @param authorityID a number of vector of numbers representing the ID(s) of the authorities (sources) which you want 
#' to use to map this gene to probes. At the time of writing, 1 = BLAST, 2 = Vendor, 3 = Bioconductor. 
#' @returnType list
#' @return a list containing information about the gene specified and a data.grame detailing the probes with which this 
#' gene is associated. There is a normalized [0-1] value representing the strength of that association according to each 
#' authority in the database. 
#' @author Jeffrey D. Allen \email{Jeffrey.Allen@@UTSouthwestern.edu}
#' @export
pm.getProbesByGene <- function (conn, entrezID, platformID, authorityID){
	
	url <- paste(conn["url"], "getProbes?",conn["keyStr"], sep="");
	
	if (missing(entrezID)){
		stop("You must provide an EntrezID");		
	}
	else{
		url <- paste(url, "&eid=",entrezID,sep="");		
	}
	
	if (!missing(authorityID)){
		authorityID <- paste(authorityID, collapse=",");
		url <- paste(url, "&authid=", authorityID, sep="");
	}
	
	if (!missing(platformID)){
		platformID <- paste(platformID, collapse=",");
		url <- paste(url, "&platid=", platformID, sep="");
	}	
	
	
	
	dat <- fromJSON(file=url);
	
	nullToNA <- function(x) sapply(x, function(x) if (is.list(x)) nullToNA(x) else if (is.null(x)) NA else x,simplify=FALSE)
	dat <- nullToNA(dat);
	
	dat$probes <- data.frame(t(sapply(dat$probes, "[")))
	
	return(dat);
	
}


#' Checks for the presence of an error in the returned JSON and prints a warning.
#' @param input the list which should be checked
#' @returnType none
#' @return none -- warns if error found
#' @author Jeffrey D. Allen \email{Jeffrey.Allen@@UTSouthwestern.edu}
checkPMError <- function (input){  
	if (any(tolower(names(input)) == "error")){
		warning(input[["error"]]);
	}
} 


