% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cluster_groups.R
\name{cluster_groups}
\alias{cluster_groups}
\title{Move data points into clusters}
\usage{
cluster_groups(
  data,
  cols,
  group_cols = NULL,
  scale_min_fn = function(x) {
     quantile(x, 0.025)
 },
  scale_max_fn = function(x) {
     quantile(x, 0.975)
 },
  keep_centroids = FALSE,
  multiplier = 0.05,
  suffix = "_clustered",
  keep_original = TRUE,
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame}. If \code{`group_cols`} is \code{NULL}, it must be grouped with
\code{\link[dplyr:group_by]{dplyr::group_by()}}.}

\item{cols}{Names of columns in \code{`data`} to mutate.
Each column is considered a dimension to contract distances in.}

\item{group_cols}{Names of grouping columns in \code{`data`}. Must be distinct from the names in \code{`cols`}.

If \code{NULL} and \code{`data`} is grouped, those groups are used instead.}

\item{scale_min_fn, scale_max_fn}{Function to find the minimum/maximum value in
the original data when rescaling the contracted data.

\strong{Input}: A \code{numeric vector}.

\strong{Output}: A \code{numeric scalar}.}

\item{keep_centroids}{Whether to ensure the clusters have their original centroids. (Logical)}

\item{multiplier}{Numeric constant to multiply the distance to the group centroid by. A smaller value
makes the clusters more compact and vice versa.}

\item{suffix}{Suffix to add to the names of the generated columns.

Use an empty string (i.e. \code{""}) to overwrite the original columns.}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with the clustered columns.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Transform values such that the elements in each group move closer to their centroid.
}
\details{
\itemize{
\item{Contracts the distance from each data point to the centroid of its group.}
\item{Performs MinMax scaling such that the scale of the data points is \emph{similar} to the original data.}
\item{If enabled (not default), the centroids are moved to the original centroids.}
}
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
library(ggplot2)

# Set seed
set.seed(2)

# Create a data frame
df <- data.frame(
  "x" = runif(50),
  "y" = runif(50),
  "z" = runif(50),
  "g" = rep(c(1, 2, 3, 4, 5), each = 10)
)

# Move the data points into clusters
cluster_groups(df,
  cols = c("x", "y"),
  group_col = "g"
)
cluster_groups(df,
  cols = c("x", "y"),
  group_col = "g",
  multiplier = 0.1
)
cluster_groups(df,
  cols = c("x"),
  group_col = "g",
  multiplier = 0.1
)

#
# Plotting clusters
#

# Cluster x and y for each group in g
df_clustered <- cluster_groups(
  data = df,
  cols = c("x", "y"),
  group_col = "g"
)

# Plot the clusters over the original data points
# As we work with random data, the cluster might overlap
ggplot(
  df_clustered,
  aes(x = x_clustered, y = y_clustered, color = factor(g))
) +
  # Original data
  geom_point(aes(x = x, y = y), alpha = 0.3, size = 0.8) +
  # Clustered data
  geom_point() +
  theme_minimal() +
  labs(x = "x", y = "y", color = "g")

#
# Maintain original group centroids
#

df_clustered <- cluster_groups(
  data = df,
  cols = c("x", "y"),
  group_col = "g",
  keep_centroids = TRUE
)

# Plot the clusters over the original data points
# As we work with random data, the cluster might overlap
ggplot(
  df_clustered,
  aes(x = x_clustered, y = y_clustered, color = factor(g))
) +
  # Original data
  geom_point(aes(x = x, y = y), alpha = 0.3, size = 0.8) +
  # Clustered data
  geom_point() +
  theme_minimal() +
  labs(x = "x", y = "y", color = "g")

#
# Three dimensions
#

# Cluster in 3d
df_clustered <- cluster_groups(
  data = df,
  cols = c("x", "y", "z"),
  group_col = "g"
)

\dontrun{
# Plot 3d with plotly
plotly::plot_ly(
  x = df_clustered$x_clustered,
  y = df_clustered$y_clustered,
  z = df_clustered$z_clustered,
  type = "scatter3d",
  mode = "markers",
  color = df_clustered$g
)
}
}
\seealso{
Other mutate functions: 
\code{\link{apply_transformation_matrix}()},
\code{\link{dim_values}()},
\code{\link{expand_distances_each}()},
\code{\link{expand_distances}()},
\code{\link{flip_values}()},
\code{\link{roll_values}()},
\code{\link{rotate_2d}()},
\code{\link{rotate_3d}()},
\code{\link{shear_2d}()},
\code{\link{shear_3d}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}

Other clustering functions: 
\code{\link{generate_clusters}()},
\code{\link{transfer_centroids}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{clustering functions}
\concept{mutate functions}
