% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{rateReg}
\alias{rateReg}
\title{Recurrent Events Regression Based on Counts and Rate Function}
\usage{
rateReg(formula, data, subset, df = NULL, knots = NULL,
        degree = 0L, na.action, spline = c("bSplines", "mSplines"),
        start = list(), control = list(), contrasts = NULL, ...)
}
\arguments{
\item{formula}{\code{Survr} object produced by function \code{\link{Survr}}.}

\item{data}{An optional data frame, list or environment containing
the variables in the model.  If not found in data, the variables are taken
from \code{environment(formula)}, usually the environment from which
function \code{\link{rateReg}} is called.}

\item{subset}{An optional vector specifying a subset of observations
to be used in the fitting process.}

\item{df}{An optional nonnegative integer to specify the degree of freedom
of baseline rate function. If argument \code{knots} or \code{degree} are
specified, \code{df} will be neglected whether it is specified or not.}

\item{knots}{An optional numeric vector that represents all the internal
knots of baseline rate function.
The default is \code{NULL}, representing no any internal knots.}

\item{degree}{An optional nonnegative integer to specify the degree of
spline bases.}

\item{na.action}{A function that indicates what should the procedure
do if the data contains \code{NA}s.  The default is set by the
na.action setting of \code{\link[base]{options}}.
The "factory-fresh" default is \code{\link[stats]{na.omit}}.
Other possible values inlcude \code{\link{na.fail}},
\code{\link{na.exclude}}, and \code{\link{na.pass}}.
\code{help(na.fail)} for details.}

\item{spline}{An optional character that specifies the flavor of splines.
The possible option is \code{bSplines} for B-splines or
\code{mSplines} for M-splines. Partial matching on the names is allowed.}

\item{start}{An optional list of starting values for the parameters
to be estimated in the model.  See more in Section details.}

\item{control}{An optional list of parameters to control the
maximization process of negative log likelihood function
and adjust the baseline rate function.
See more in Section details.}

\item{contrasts}{An optional list, whose entries are values
(numeric matrices or character strings naming functions) to be used
as replacement values for the contrasts replacement function and
whose names are the names of columns of data containing factors.
See \code{contrasts.arg} of \code{\link[stats]{model.matrix.default}}
for details.}

\item{...}{Other arguments for future usage.}
}
\value{
A \code{\link{rateReg-class}} object, whose slots include
\itemize{
    \item \code{call}: Function call of \code{rateReg}.
    \item \code{formula}: Formula used in the model fitting.
    \item \code{nObs}: Number of observations.
    \item \code{spline}: A list contains
        \itemize{
            \item \code{spline}: The name of splines used.
            \item \code{knots}: Internal knots specified for the baseline
                rate function.
            \item \code{Boundary.knots}: Boundary knots specified for the
                baseline rate function.
            \item \code{degree}: Degree of spline bases specified in
                baseline rate function.
            \item \code{df}: Degree of freedom of the model specified.
    }
    \item \code{estimates}: Estimated coefficients of covariates and
        baseline rate function, and estimated rate parameter of
        gamma frailty variable.
    \item \code{control}: The control list specified for model fitting.
    \item \code{start}: The initial guess specified for the parameters
        to be estimated.
    \item \code{na.action}: The procedure specified to deal with
        missing values in the covariate.
    \item \code{xlevels}: A list that records the levels in
        each factor variable.
    \item \code{contrasts}: Contrasts specified and used for each
        factor variable.
    \item \code{convergCode}: \code{code} returned by function
        \code{\link[stats]{nlm}}, which is an integer indicating why the
        optimization process terminated. \code{help(nlm)} for details.
    \item \code{logL}: Log likelihood of the fitted model.
    \item \code{fisher}: Observed Fisher information matrix.
}
}
\description{
The default model is the gamma frailty model with one piece constant
baseline rate function, which is equivalent to negative binomial regression
with the same shape and rate parameter in the gamma prior. Spline (including
piecewise constant) baseline rate function can also be specified and applied
to model fitting.  Both B-spline and M-spline bases are available.
\code{rateReg} returns the fitted model through a
\code{\link{rateReg-class}} object.
}
\details{
Function \code{\link{Survr}} in the formula response by default first checks
the dataset and will report an error if the dataset does not fall into
recurrent event data framework.  Subject's ID will be pinpointed if its
observation violates any checking rule. See \code{\link{Survr}} for all the
checking rules.

Function \code{rateReg} first constructs the design matrix from
the specified arguments: \code{formula}, \code{data}, \code{subset},
\code{na.action} and \code{constrasts} before model fitting.
The constructed design matrix will be checked again to
fit the recurrent event data framework
if any observation with missing covariates is removed.

The model fitting process involves minimization of negative log
likelihood function, which calls function \code{\link[stats]{nlm}}
from package \pkg{stats} internally.
\code{help(nlm)} for more details.

The argument \code{start} is an optional list
that allows users to specify the initial guess for
the parameter values for the minimization of
negative log likelihood function.
The available numeric vector elements in the list include
\itemize{
    \item \code{beta}: Coefficient(s) of covariates,
        set to be 0.1 by default.
    \item \code{theta}: Parameter in Gamma(theta, 1 / theta) for
        frailty random effect, set to be 0.5 by default.
    \item \code{alpha}: Coefficient(s) of baseline rate function,
        set to be 0.05 by default.
}
The argument \code{control} is an optional list
that allows users to control the process of minimization of
negative log likelihood function and to specify the boundary knots,
intercept for baseline rate function.
The available elements in the list include
\itemize{
    \item \code{gradtol}: A positive scalar giving the tolerance at
        which the scaled gradient is considered close enough to zero
        to terminate the algorithm. The default value is 1e-6.
    \item \code{stepmax}: A positive scalar that gives the maximum
        allowable scaled step length. The default value is 1e5.
    \item \code{steptol}: A positive scalar providing the minimum
        allowable relative step length. The default value is 1e-6.
    \item \code{iterlim}: A positive integer specifying the maximum
        number of iterations to be performed before
        the program is terminated. The default value is 1e2.
    \item \code{Boundary.knots}: A length-two numeric vector to specify
        the boundary knots for baseline rate funtion. By default,
        the left boundary knot is zero and the right one takes the
        largest censoring time from data.
    \item \code{verbose}: A optional logical value with default \code{TRUE}.
        Set it to be \code{FALSE} to supress any possible message
        from this function.
}
}
\examples{
library(reda)

## constant rate function
(constFit <- rateReg(Survr(ID, time, event) ~ group + x1, data = simuDat))

## six pieces' piecewise constant rate function
(piecesFit <- rateReg(Survr(ID, time, event) ~ group + x1,
                      data = simuDat, subset = ID \%in\% 1:50,
                      spline = "bSplines", knots = seq(28, 140, by = 28)))

## fit rate function with cubic spline
(splineFit <- rateReg(Survr(ID, time, event) ~ group + x1, data = simuDat,
                      spline = "mSpl", knots = c(56, 84, 112), degree = 3))

## more specific summary
summary(constFit)
summary(piecesFit)
summary(splineFit)

## model selection based on AIC or BIC
AIC(constFit, piecesFit, splineFit)
BIC(constFit, piecesFit, splineFit)

## estimated covariate coefficients
coef(piecesFit)
coef(splineFit)

## confidence intervals for covariate coefficients
confint(piecesFit)
confint(splineFit, "x1", 0.9)
confint(splineFit, 1, 0.975)

## estimated coefficients for baseline rate function
baseRate(piecesFit)
baseRate(splineFit)

## estimated baseline mean cumulative function (MCF) from a fitted model
piecesMcf <- mcf(piecesFit)
plot(piecesMcf, conf.int = TRUE, col = "blueviolet")

## estimated MCF for given new data
newDat <- data.frame(x1 = rep(0, 2), group = c("Treat", "Contr"))
splineMcf <- mcf(splineFit, newdata = newDat, groupName = "Group",
                 groupLevels = c("Treatment", "Control"))
plot(splineMcf, conf.int = TRUE, lty = c(1, 5)) + ggplot2::xlab("Days")
}
\references{
Fu, H., Luo, J., & Qu, Y. (2016).
Hypoglycemic events analysis via recurrent time-to-event (HEART) models.
\emph{Journal Of Biopharmaceutical Statistics}, 26(2), 280--298.
}
\seealso{
\code{\link{summary,rateReg-method}} for summary of fitted model;
\code{\link{coef,rateReg-method}} for estimated covariate coefficients;
\code{\link{confint,rateReg-method}} for confidence interval of
covariate coefficients;
\code{\link{baseRate,rateReg-method}} for estimated coefficients of baseline
rate function;
\code{\link{mcf,rateReg-method}} for estimated MCF from a fitted model;
\code{\link{plot,rateRegMcf-method}} for plotting estimated MCF.
}

