##' Assists creation of predicted value curves for regression models. 
##' 
##'
##' This is similar to \code{plotSlopes}, but it accepts regressions
##' in which there are transformed variables, such as "log(x1)".
##' Think of this a new version of R's \code{termplot}, but it allows
##' for interactions.  It creates a plot of the predicted dependent
##' variable against one of the numeric predictors, \code{plotx}. It
##' draws a predicted value line for several values of \code{modx}, a
##' moderator variable. The moderator may be a numeric or categorical
##' moderator variable.
##'
##' The user may designate which particular values of the moderator
##' are used for calculating the predicted value lines.   That is,
##' \code{modxVals = c( 12,22,37)} would draw lines for values 12, 22,
##' and 37 of the moderator.
##'
##' If the user does not specify the parameter \code{modxVals},
##' built-in algorithms will select the "cut points". Three algorithms
##' have been prepared so far, \code{quantile}, \code{std.dev.}, and
##' \code{table}. If the number of unique observed values is smaller
##' than 6, the \code{table} method is used.  The 5 most frequently
##' observed values of modx are selected. Otherwise, the quantile
##' method is used. Predictive lines are plotted for the following
##' percentiles {0.25,0.50,0.75}. The algorithm \code{std.dev.} plots
##' three lines, one for the mean of modx, and one for the mean minus
##' one standard deviation, and the other for the mean plus one
##' standard deviation. 
##' 
##' 
##' @param model Fitted regression object. Must have a predict method
##' @param plotx String with name of IV to be plotted on x axis
##' @param modx String for moderator variable name. May be either numeric or factor.
##' @param modxVals If modx is numeric, either a character string, "quantile", "std.dev.",
##' or "table", or a vector of values for which plotted lines are
##' sought. If modx is a factor, the default approach will create one
##' line for each level, but the user can supply a vector of levels if
##' a subset is desired.
##' @param plotPoints Should the plot include the scatterplot points along with the lines.
##' @param envir environment to search for variables. 
##' @param ... further arguments that are passed to plot.
##' @export
##' @import car
##' @return A plot is created as a side effect, a list is returned including
##' 1) the call, 2) a newdata object that includes information on the curves that were 
##' plotted, 3) a vector modxVals, the values for which curves were drawn.
##' @author Paul E. Johnson <pauljohn@@ku.edu>
##' @example  inst/examples/plotCurves-ex.R

plotCurves <-
  function (model = NULL, plotx = NULL, modx = NULL, modxVals = NULL, 
            plotPoints = TRUE, envir = environment(formula(model)), ...) 
{
  if (is.null(model)) 
    stop("plotCurves requires a fitted regression model.")
  if (is.null(plotx)) 
    stop("plotCurves requires the name of the variable to be drawn on the x axis")
  if (is.null(modx)) 
    stop("plotCurves requires the name of moderator variable for which several slopes are to be drawn")

  carrier <- function(term, data, enc = NULL) {
    if (length(term) > 1L) 
      carrier(term[[2L]])
    else eval(term, envir = data, enclos = enc)
  }
  carrier.name <- function(term) {
    if (length(term) > 1L) 
      carrier.name(term[[2L]])
    else as.character(term)
  }

  cutByTable <- function(x, n = 5){
    table1 <- table(x)
    table1sort <-  sort(table1, decreasing = T)
    qs <- table1sort[1:n]
    names(qs) <- names(table1sort[1:n])
    invisible(qs)
  }
  
  cutByQuantile <- function(x){
    uniqueVals <- unique(x)
    if (length(uniqueVals) < 6) {
      qs <- cutByTable(x, 5)
      invisible(qs)
    } else {
      qs <- quantile(x, probs = c(0.25, 0.50, 0.75), na.rm = TRUE)
      invisible(qs)
    }
  }
  
  cutBySD <- function(x){
    uniqueVals <- unique(x)
    if (length(uniqueVals) < 6) {
      qs <- cutByTable(x, 5)
      invisible(qs)
    } else {
      mx <- round(mean(x, na.rm=T),2)
      sdx <- round(sd(x, na.rm=T),2)
      ##qs <- c(mx - 2*sdx, mx - sdx, mx, mx + sdx, mx + 2*sdx)
      ##suffix <- c("(m-2sd)","(m-sd)","(m)","(m+sd)","(m+2sd)")
      qs <- c(mx - sdx, mx, mx + sdx)
      suffix <- c("(m-sd)","(m)","(m+sd)")
      names(qs) <-  paste(qs, suffix)
      invisible(qs)
    }
  }

  
  cl <- match.call()
  mf <- model.frame(model)
  tt <- terms(model)
    
  cn <- parse(text = colnames(mf))
  varnames <- unlist(lapply(cn, carrier.name))

  emf <- get_all_vars(tt, data = expand.model.frame(model, varnames, na.expand=TRUE))

  ## experimenting with another way to gather variables.
  ## data <- eval(model$call$data, envir) ##grabs nothing unless data option was used
  ## if (is.null(data)) 
  ##   data <- mf
  ## ## if (plotx %in% varnames) plotxVar <- emf[, plotx] else stop("plotx missing")
  ## data <- data[row.names(emf) , ]
  
  plotxVar <- carrier(parse(text = plotx), emf, enc=envir)
  if (!is.numeric(plotxVar)) 
    stop(paste("plotCurves: The variable", plotx, "should be a numeric variable"))

  modxVar <- carrier(parse(text = modx), emf, enc=envir)
  depVar <- model.response(mf)
 
  ylab <- names(mf)[1]  ## returns transformed DV
  ##ylab <- varnames[1] ## returns untransformed carrier DV
  plotyRange <- magRange(depVar, mult=c(1,1.2))
  plotxRange <- range(plotxVar, na.rm=TRUE)
  plotxSeq <- plotSeq(plotxRange, length.out = 40)

  if (is.factor(modxVar)) { ## modxVar is a factor
    if (is.null(modxVals)) {
      modxVals <- levels(modxVar)
    } else {
      if (!all(modxVals %in% levels(modxVar))) stop("modxVals includes non-observed levels of modxVar")
    }
  } else {                  ## modxVar is not a factor
    modxRange <- range(modxVar, na.rm=TRUE)
    if (is.null(modxVals)) {
      modxVals <- cutByQuantile(modxVar)
    } else {
      if (is.numeric(modxVals)) { 
      ##TODO: Insert some checks that modxVals are reasonable
      } else {
        if (is.character(modxVals)) {
          modxVals <- match.arg(tolower(modxVals),
                                c("quantile", "std.dev.", "table"))
          print(modxVals)
          modxVals <- switch(modxVals,
                             table = cutByTable(modxVar),
                             quantile = cutByQuantile(modxVar),
                             "std.dev." = cutBySD(modxVar),
                             stop("unknown 'modxVals' algorithm"))
        }
      }
    }
  }
  lmx <- length(modxVals)                            
  
  predictors <- colnames(emf)[-1]
  predictors <- setdiff(predictors, c(modx, plotx))
  newdf <- data.frame(expand.grid(plotxSeq, modxVals))
  colnames(newdf) <- c(plotx, modx)
  if (length(predictors) > 0) {
    newdf <- cbind(newdf, centralValues(as.data.frame(emf[, predictors])))
    colnames(newdf) <- c(plotx, modx, predictors)
  }
  newdf$pred <- predict(model, newdata = newdf)
  dotargs <- list(...)
  if (!plotPoints){
    parms <- list(plotxVar, depVar, xlab = plotx, ylab = ylab, ylim = plotyRange,
         type = "n")
    parms <- modifyList(parms, dotargs)
    do.call("plot", parms)
  } else {
    if (is.factor(modxVar)) {
      parms <- list(plotxVar, depVar, xlab = plotx, ylab = ylab, ylim = plotyRange,
           col = modxVar)
      parms <- modifyList(parms, dotargs)
      do.call("plot", parms)
    }
    else {
      parms <- list(plotxVar, depVar, xlab = plotx, ylab = ylab, ylim = plotyRange)
      parms <- modifyList(parms, dotargs)
      do.call("plot", parms)
    }
  }
  for (i in 1:lmx) {
    pdat <- newdf[newdf[, modx] %in% modxVals[i], ]
    lines(pdat[, plotx], pdat$pred, lty = i, col = i, lwd = 2)
  }
  if (is.null(names(modxVals))) {
    legnd <- paste(modxVals, sep = "")
  }
  else {
    legnd <- paste(names(modxVals), sep = "")
  }
  legend("topleft", legend = legnd, lty = 1:lmx, col = 1:lmx, 
         bg = "white", title= paste("moderator:", modx))

  invisible(list(call=cl, newdata=newdf, modxVals = modxVals))
}
