

#' Trim Zero Rows
#'
#' @param tbl table object
#'
#' @return an rtable object
#'
#' @export
#'
trim_zero_rows <- function(tbl) {
  stopifnot(is(tbl, "VTableTree"))

  rows <- collect_leaves(tbl, TRUE, TRUE)
  torm <- vapply(rows, function(x) {
    identical(unname(unlist(row_values(x))), rep(0L, ncol(tbl)))
  }, NA, USE.NAMES = FALSE)
  tbl[!torm, , keep_topleft = TRUE ]

}



#' Score functions for sorting TableTrees
#' @rdname score_funs
#' @inheritParams gen_args
#' @return A single numeric value indicating score according to the relevant metric for \code{tt}, to be used
#' when sorting.
#' @export
cont_n_allcols <- function(tt) {
    ctab <- content_table(tt)
    if(NROW(ctab) == 0)
        return(NA)

    sum(sapply(row_values(tree_children(ctab)[[1]]),
               function(cv) cv[1]))
}

#' @rdname score_funs
#' @param j numeric(1). Number of column to be scored
#' @export
cont_n_onecol <- function(j) {
    function(tt) {
        ctab <- content_table(tt)
        if(NROW(ctab) == 0)
            return(NA)
        row_values(tree_children(ctab)[[1]])[[j]][1]
    }
}

#' Sort substructure of a TableTree at a particular Path in the Tree.
#' @inheritParams gen_args
#' @param scorefun function. Scoring function, should accept the type of children directly under the position at \code{path} (either VTableTree, VTableRow, or VTableNodeInfo, which covers both) and return a numeric value to be sorted.
#' @param decreasing logical(1). Should the the scores generated by \code{scorefun} be sorted in decreasing order. If unset (the default of \code{NA}), it is set to \code{TRUE} if the generated scores are numeric and \code{FALSE} if they are characters.
#' @param na.pos character(1). What should be done with children (subtrees/rows) with \code{NA} scores. Defaults to \code{"omit"}, which removes them, other allowed values are \code{"last"}  and \code{"first"} which indicate where they should be placed in the order.
#' @return A TableTree with the same structure as \code{tt} with the exception that the requested sorting has been done at \code{path}
#' @details
#' The \code{path} here can include \code{"*"} as a step, which means taht each child at that step will be \emph{separately} sorted based on \code{scorefun} and the remaining \code{path} entries. This can occur multiple times in a path.
#'
#'
#' @export
sort_at_path <- function(tt, path, scorefun, decreasing = NA, na.pos = c("omit", "last", "first")) {
    if(NROW(tt) == 0)
        return(tt)

    ## XXX hacky fix this!!!
    if(identical(obj_name(tt), path[1]))
        path <- path[-1]

    curpath <- path
    subtree <- tt
    backpath <- c()
    while(length(curpath) > 0) {
        curname <- curpath[1]
        ## we sort each child separately based on the score function
        ## and the remaining path
        if(curname == "*") {
            newkids = lapply(tree_children(subtree),
                             sort_at_path,
                             path = curpath[-1],
                             scorefun = scorefun,
                             decreasing = decreasing,
                             na.pos = na.pos)
            newtab <- subtree
            tree_children(newtab) <- newkids
            if(length(backpath) > 0) {
                ret <- recursive_replace(tt, backpath, value = newtab)
            } else {
                ret <- newtab
            }
            return(ret)
        }
        subtree <- tree_children(subtree)[[curname]]
        backpath <- c(backpath, curpath[1])
        curpath <- curpath[-1]
    }

    na.pos <- match.arg(na.pos)
##    subtree <- tt_at_path(tt, path)
    kids <- tree_children(subtree)
    ## relax this to allow character "scores"
    ## scores <- vapply(kids, scorefun, NA_real_)
    scores <- sapply(kids, scorefun)
    if(!is.null(dim(scores)) ||
       length(scores) != length(kids))
        stop("Score function does not appear to have return exactly one scalar value per child")
    if(is.na(decreasing))
        decreasing <- if(is.character(scores)) FALSE else TRUE
    ord <- order(scores, na.last = (na.pos != "first"), decreasing = decreasing)
    newkids <- kids[ord]
    if(anyNA(scores) && na.pos == "omit") { #we did na last here
        newkids = head(newkids, -1*sum(is.na(scores)))
    }

    newtree = subtree
    tree_children(newtree) <- newkids
    tt_at_path(tt, path) <- newtree
    tt
}
