% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{crankNicolson1D}
\alias{crankNicolson1D}
\title{Crank--Nicolson finite difference scheme for the 1D Fokker--Planck equation with periodic boundaries}
\usage{
crankNicolson1D(u0, b, sigma2, N, deltat, Mx, deltax, imposePositive = 0L)
}
\arguments{
\item{u0}{matrix of size \code{c(Mx, 1)} giving the initial condition. Typically, the evaluation of a density highly concentrated at a given point. If \code{nt == 1}, then \code{u0} can be a matrix \code{c(Mx, nu0)} containing different starting values in the columns.}

\item{b}{vector of length \code{Mx} containing the evaluation of the drift.}

\item{sigma2}{vector of length \code{Mx} containing the evaluation of the squared diffusion coefficient.}

\item{N}{increasing integer vector of length \code{nt} giving the indexes of the times at which the solution is desired. The times of the solution are \code{delta * c(0:max(N))[N + 1]}.}

\item{deltat}{time step.}

\item{Mx}{size of the equispaced spatial grid in \eqn{[-\pi,\pi)}.}

\item{deltax}{space grid discretization.}

\item{imposePositive}{flag to indicate whether the solution should be transformed in order to be always larger than a given tolerance. This prevents spurious negative values. The tolerance will be taken as \code{imposePositiveTol} if this is different from \code{FALSE} or \code{0}.}
}
\value{
\itemize{
\item If \code{nt > 1}, a matrix of size \code{c(Mx, nt)} containing the discretized solution at the required times.
\item If \code{nt == 1}, a matrix of size \code{c(Mx, nu0)} containing the discretized solution at a fixed time for different starting values.
}
}
\description{
Implementation of the Crank--Nicolson scheme for solving the Fokker--Planck equation
\deqn{p(x, t)_t = -(p(x, t) b(x))_x + \frac{1}{2}(\sigma^2(x) p(x, t))_{xx},}{p(x, t)_t = -(p(x, t) * b(x))_x + 1/2 * (\sigma^2(x) p(x, t))_{xx},}
where \eqn{p(x, t)} is the transition probability density of the circular diffusion
\deqn{dX_t=b(X_t)dt+\sigma(X_t)dW_t}{dX_t=b(X_t)dt+\sigma(X_t)dW_t}.
}
\details{
The function makes use of \code{\link{solvePeriodicTridiag}} for obtaining implicitly the next step in time of the solution.

If \code{imposePositive = TRUE}, the code implicitly assumes that the solution integrates to one at any step. This might b unrealistic if the initial condition is not properly represented in the grid (for example, highly concentrated density in a sparse grid).
}
\examples{
# Parameters
Mx <- 200
N <- 200
x <- seq(-pi, pi, l = Mx + 1)[-c(Mx + 1)]
times <- seq(0, 1, l = N + 1)
u0 <- dWn1D(x, pi/2, 0.05)
b <- driftWn1D(x, alpha = 1, mu = pi, sigma = 1)
sigma2 <- rep(1, Mx)

# Full trajectory of the solution (including initial condition)
u <- crankNicolson1D(u0 = cbind(u0), b = b, sigma2 = sigma2, N = 0:N,
                     deltat = 1 / N, Mx = Mx, deltax = 2 * pi / Mx)

# Mass conservation
colMeans(u) * 2 * pi

# Visualization of tpd
plotSurface2D(times, x, z = t(u), levels = seq(0, 3, l = 50))

# Only final time
v <- crankNicolson1D(u0 = cbind(u0), b = b, sigma2 = sigma2, N = N,
                     deltat = 1 / N, Mx = Mx, deltax = 2 * pi / Mx)
sum(abs(u[, N + 1] - v))
}
\references{
Thomas, J. W. (1995). \emph{Numerical Partial Differential Equations: Finite Difference Methods}. Springer, New York. \doi{10.1007/978-1-4899-7278-1}
}
