# subfunctions that search output files for some information
# used by: seas

detect_error <- function(err){
  # error parsing from .err or .err.html character vector
  #
  # err  character vector, content of output file
  #
  # returns an object of class x13messages which can be printed
  
  if (getOption("htmlmode") == 1){
    ParseInfo <- function(openl, x){
      # find next closing tag
      clt <- grep("</p>", x)
      closel <- clt[clt >= openl][1]

      # extract info between tags
      z <- paste(x[openl:closel], collapse = "")

      # clean info
      z <- gsub("<p>.*</strong>", "", z) # remove trailing tag
      z <- gsub("</p>", "", z)           # remove closing tag 
      z <- gsub("&nbsp;", "", z)  
      z <- gsub("\\s+", " ", z)          # remove multiple space
      z <- gsub("^\\s", "", z)           # remove trailing space
      z <- gsub("<.+?>", "", z)           # remove inside HTML tags
      z
    }
  } else {
    ParseInfo <- function(openl, x){
      line2 <- NULL
      for (l in openl:length(x)){
        if (x[l] == "  "){
          line2 <- l - 1
        }
      }
      if (is.null(line2)){
        line2 <- length(x)
      }
      z <- paste(x[openl:line2], collapse = "")
      z <- gsub("^.*: ", "", z)    # remove trailing tag
      z <- gsub("^\\s", "", z)     # remove trailing space
    }
  }

  z <- list()
  class(z) <- "x13messages"
  z$error <- sapply(grep("ERROR:", err), ParseInfo, x = err)
  z$warning <- sapply(grep("WARNING:", err), ParseInfo, x = err)
  # do not show this meaningless warning 
  # (its caused by default activation of spectrum)
  z$warning <- z$warning[!grepl("Spectrums are only generated for monthly series.", z$warning)]
  z$note <- sapply(grep("note:", err), ParseInfo, x = err)
  z
}


print.x13messages <- function(x){
  if (length(x$error) == 0 & length(x$warning) == 0 & length(x$note) == 0 ){
    return(NULL)
  }
  cat("\nMessages generated by X-13:\n")
  if (length(x$error) > 0){
    cat("Errors:\n")
    cat((paste(strwrap(paste("-", x$error), width = 60, exdent = 2), 
               collapse = "\n")))
    cat("\n")
  }
  if (length(x$warning) > 0){
    cat("Warnings:\n")
    cat((paste(strwrap(paste("-", x$warning), width = 60, exdent = 2), 
               collapse = "\n")))
    cat("\n")
  }
  if (length(x$note) > 0){
    cat("Notes:\n")
    cat((paste(strwrap(paste("-", x$note), width = 60, exdent = 2), 
               collapse = "\n")))
    cat("\n")
  }
}


detect_auto <- function(outtxt){
  # parse automatic log detection from .out txt
  #
  # outtxt  character vector, content of .out output file
  #
  # returns character string, "log" or "none"
  
  if (getOption("htmlmode") == 1){
    first <- which(outtxt == "<h3> Likelihood statistics for model fit to log transformed series.</h3>")
  } else {
    first <- which(outtxt == " Likelihood statistics for model fit to log transformed series.")
  }
  
  if (length(first) == 0){
    return(NULL)
  }
  area <- outtxt[(first + 10):(first + 30)]
  line <- area[grepl("prefers", area)]
  
  if (length(line) == 1){
    if (grepl("log", line)){
      z <- "log"
    } else {
      z <- "none"
    }
  } else {
    z <- NULL
  }
  z
}


detect_fivebestmdl <- function(outtxt){
  # parse fivebestmdl from .out txt
  #
  # outtxt  character vector, content of .out output file
  #
  # returns character vector
  
  if (getOption("htmlmode") == 1){
    first <- which(outtxt == "<p>Best Five ARIMA Models</p>")
  } else {
    first <- which(outtxt == "  Best Five ARIMA Models")
  }
  
  # if there are several runs, take the last
  if (length(first) >= 1) {
    z <- outtxt[first[length(first)]:(first[length(first)] + 10)]
  } else {
    z <- NULL
  }
  z
}


detect_qs <- function(outtxt){
  # parse QS test for seasonality from .out txt
  #
  # outtxt  character vector, content of .out output file
  #
  # returns character vector
  
  if (getOption("htmlmode") == 1){
    first <- which(outtxt == "<caption><strong>QS statistic for seasonality (Full series)</strong></caption>")
    # lines to show (-1)
    nl <- grep("Irregular Series \\(extreme value adjusted\\)", outtxt[first:(first + 20)])
    txt <- outtxt[(first + 8):(first + nl - 1)]
    # parse fixed width table
    descr <- substr(txt, start = 21, stop = 51)
    descr <- gsub("^\\s+|\\s+$", "", descr)   # trim lead. and trail spaces
    stat <- as.numeric(substr(txt, start = 52, stop = 60))
    pval <- as.numeric(substr(txt, start = 71, stop = 81))
    
    
  } else {
    first <- which(outtxt == "  QS statistic for seasonality:")
  }
  if (length(first) == 1){
    # lines to show (-1)
    nl <- grep("Irregular Series \\(EV adj\\)", outtxt[first:(first + 10)])
    txt <- outtxt[(first + 1):(first + nl - 1)]
    # parse fixed width table
    descr <- substr(txt, start = 3, stop = 51)
    descr <- gsub("^\\s+|\\s+$", "", descr)   # trim lead. and trail spaces
    stat <- as.numeric(substr(txt, start = 52, stop = 60))
    pval <- as.numeric(substr(txt, start = 71, stop = 81))
    
    z <- cbind(stat, pval)
    rownames(z) <- descr
    z <- z[!is.na(z[, "stat"]), ]
    
  } else {
    z <- NULL
  }
  z
}


