\name{Model eval and sensitivity}
\alias{model.eval}
\alias{sens}
\alias{msens}
\alias{mway.fact}
\alias{mway.samp}

\title{Model evaluation and sensitivity analysis}

\description{
Function model.eval provides calculation of errors between observed or reference values and simulated values.

Function sens provides sensitivity analysis on one parameter using six metrics. Function msens conducts sensitivity analysis on combinations of several parameters. These are combined by full-factorial (using mway.fact) or stratified sampling by Latin Hyper-cubes (using mwy.samp).

}
\usage{
model.eval(t, Xobs, Xmod, fileout, pdfout=F)
sens(t.X, param, fileout, pdfout=F)
msens(t.X, param, fileout, pdfout=F, resp=F)
mway.fact(pnom,plab,perc.v,levels)
mway.samp(pnom,plab,perc.v,runs)
}

\arguments{
  \item{t}{times to evaluate output}
  \item{Xobs}{observed values}
  \item{Xmod}{simulated values}
  \item{fileout}{name of output file}
  \item{pdfout}{logical to decide to produce PDF output file}
  \item{t.X}{output of multiple runs for sensitivity}
  \item{param}{list of parameter variation for sensitivity, nominal, label, values, and a logical variable fact for factorial or Monte Carlo analysis}
  \item{resp}{logical variable to decide upon producing values of response function}
  \item{pnom}{array of nominal parameter values}
  \item{plab}{array of parameter labels}
  \item{perc.v}{percent variation around nominal}
  \item{levels}{levels for full factorial analysis}
  \item{runs}{number of runs for Monte Carlo}
}
\details{
Function model.eval calculates absolute, relative and square errors or each pair of observed and simulated data. It also provides total absolute, relative, and RMS.

Function sens operates on output from multiple runs made by varying a parameter and examining the change of one or several metrics of the output. These all apply to X(t) for the entire simulation run; that is, a metric summarizes the entire run in just one value. Therefore, we can analyze the effect of changing the parameter on the value of the metric and plot x-y graphs for visualization.

The six metrics are: g1() is the maximum value of Xi for run i; g2() is the minimum value of Xi for run i; g3() is the value of Xi for run i; g4() is the average of the difference Xi-Xn for run i; g5() is the RMS of the difference Xi-Xn for run i; and g6() is the maximum of the absolute value of the difference Xi-Xn for run i. Note that the last two metrics always yield positive values. Their sign is inverted for values corresponding to the parameter values below the nominal.

For function sens: often, we perform an odd number of runs where the nominal value corresponds to the center value in the set of parameter values, and we vary by a percent. However, the nominal can be anywhere as dictated by the component pnom in the param argument.

Function msens: The pairwise combinations for more than two parameters employ the function factorial and the function tapply. The latter applies the mean to the response using the parameters as factors and produces tables x and xp, which are the basis to calculate the effects and maps of the response. The average effects are computed from the tables xp. Multiple regression is performed using lm for each metric versus the relative parameter values, ps, which had been previously standardized so that we can compare coefficients across the parameters. We extract the coefficients and the R2 from the summary of lm, which is a list. Then, we use the coefficients to calculate the predicted metrics in order to employ these later to graph the scatter plots of metric calculated versus metric predicted. Finally, all information is packed in a data.frame to be returned by the function. Interaction plots are produced using the function interaction.plots. Response maps are generated by means of the function image and the overlapping isolines are generated using the function contour with the option add=T.

The function mway-fact helps determine a full factorial design. For example, we can design the factorial of three levels at 50 percent. Function mway.fact uses two R functions, gl and factor. The output of mway.fact is used as an argument to simulation functions (e.g, sim) and to msens.

Function mway.samp is for stratified sampling of more than one parameter that varies at a time. It performs Latin hypercubes to obtain n numbers by sampling using a uniform distribution from a min to a max value determined from the desired percent variation. It is limited to the uniform distribution but can be extended to sample other distributions. The output of mway.samp is used as an argument to simulation functions (e.g, sim) and to msens.

}
\value{
For model.eval
\item{result}{Table of individual errors}
\item{label.result.tot.avg}{Labels to identify the columns of total error}
\item{result.tot.avg}{Total errors}
For sens
\item{val.val}{metric values corresponding to param values}
\item{perc.perc}{percent change in the metric to percent change in the parameter value}
\item{perc.ratio}{percent ratio change in the metric to percent change in the parameter}
\item{reg.slope.R2}{slope and R-square of regression lines for sensitivity analysis of Monte Carlo runs} 
For mway.fact
\item{plab}{array of parameter labels}
\item{pval}{combined parameter values}
\item{pnom}{array of nominal values}
\item{fact}{echo of fact argument}
\item{pv}{table of parameter values}
For mway.samp
\item{plab}{array of parameter labels}
\item{pval}{combined parameter values}
\item{pnom}{array of nominal values}
\item{fact}{echo of fact argument}
For msens
\item{Ma}{values of parameter values and six metrics for each run}
\item{Mp}{percent change of parameter and six metrics for each run}
\item{x}{When resp=T, values to make these response maps. This list has six elements, one for each metric. The first four of each set of six correspond to the metrics. The other two refer to the difference with respect to nominal.}
\item{AOV.pvalue}{when fact=T, p-values of the ANOVA to examine whether there is a significant difference in the response of the metrics}
\item{Friedman.pvalue}{when fact=T, p-values of the nonparametric test to examine whether there is a significant difference in the response of the metrics}
\item{Eff}{when fact=T, average effect for each parameter on all metrics}
\item{Eff.Int}{when fact=T, effect for each parameter in interaction with the others on all metrics} 
\item{reg.slope.R2}{When fact=F, slope and R-square of regression lines for sensitivity analysis of Monte Carlo runs} 
}

\references{
Acevedo M.F. 2012. Simulation of Ecological and Environmental Models. CRC Press.

Kirchner, T.B. 1992. TIME-ZERO The Integrated Modeling Environment. Reference Manual. Quaternary Software.

}
\author{
Miguel F. Acevedo \email{Acevedo@unt.edu}
}
\note{
Function model.eval produces a text file with errors and graphics.

For function sens: often, we perform an odd number of runs where the nominal value corresponds to the center value in the set of parameter values, and we vary by a percent. However, the nominal place can be anywhere as dictated by the component pnom in the param argument.

Input files are in 'datafiles.zip' in directory 'datafiles' and organized by chapters of Acevedo (2012). Input files are required to run the examples below.
}

\seealso{
Simulation functions \code{\link{sim.comp}}, \code{\link{sim.mruns}}
}
\examples{
\dontrun{
# model evaluation
# read observed data from file
t.Xobs <- matrix(scan("chp5/expodecay-data.txt", skip=1), ncol=2, byrow=TRUE)
# simulate
model <- list(f=expon)
# assign values for calculating errors
t.Xmod <- sim.comp(model,"chp5/exp-decay-inp.csv")
t <- t.Xobs[,1]; Xobs <- t.Xobs[,2]; Xmod <- t.Xmod$output[,2]
# run model evaluation
exp.eval <- model.eval(t, Xobs, Xmod, "chp5/eval-decay")

# sensitivity analysis on full factorial runs
# give nom and sd
pnom=0.02; psd =0.005; plab = c("r"); runs=5
# factorial
pval <- seq((pnom -(runs-1)*psd),(pnom+ (runs-1)*psd),2*psd)
param <- list(plab=plab, pval=pval, pnom=pnom, fact=TRUE)
# multiple run
t.X <- sim.mruns(model,"chp5/exp-sens-inp.csv", param)
# sensitivity
s.y <- sens(t.X$output, param, "chp5/exp-sens-plots-fact-out")

# sensitivity analysis on Monte Carlo runs 
pnom=0.02; psd =0.005; plab = c("r"); runs=10
pval <- round(c(pnom,rnorm(runs,pnom,psd)),5)
param <- list(plab=plab, pval=pval, pnom=pnom, fact=FALSE)
t.X <- sim.mruns(model,"chp5/exp-sens-inp.csv", param)
s.y <- sens(t.X$output, param, "chp5/exp-sens-plots-samp-out")

# gives nom lab, v in percent and levels
pnom= c(0.2,100); plab= c("r","K"); perc.v= 50; levels=3
param <- mway.fact(pnom,plab,perc.v,levels)

# use param output for simulator and msens
logis <- list(f=logistic)
t.X <- sim(logis,"chp8/logistic-100-inp csv",param,labout="sens-fact-")
s.y <- msens(t.X$output, param, "chp8/logis-100-sens-plots-factout",resp=FALSE)

# 10 runs by stratified sampling 
pnom= c(0.2,100); plab= c("r","K"); perc.v= 50; runs=10
param <- mway.samp(pnom,plab,perc.v,runs)

# use param output for simulator and msens
logis <- list(f=logistic)
t.X <- sim(logis,"chp8/logistic-100-inp csv",param,labout="sens-fact-")
s.y <- msens(t.X$output, param, "chp8/logis-100-sens-plots-factout",resp=FALSE)
}
}

\keyword{ math }
\keyword{ IO }
\keyword{ dplot }


