% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree_clade_phylm.R
\name{tree_clade_phylm}
\alias{tree_clade_phylm}
\title{Interaction between phylogenetic uncertainty and influential clade detection - Phylogenetic Linear Regression}
\usage{
tree_clade_phylm(formula, data, phy, clade.col, n.species = 5, n.sim = 100,
  n.tree = 2, model = "lambda", track = TRUE, ...)
}
\arguments{
\item{formula}{The model formula}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'multiPhylo', see ?\code{ape}).}

\item{clade.col}{The column in the provided data frame which specifies the
clades (a character vector with clade names).}

\item{n.species}{Minimum number of species in a clade for the clade to be
included in the leave-one-out deletion analysis. Default is \code{5}.}

\item{n.sim}{Number of simulations for the randomization test.}

\item{n.tree}{Number of times to repeat the analysis with n different trees picked 
randomly in the multiPhylo file.
If NULL, \code{n.tree} = 2}

\item{model}{The phylogenetic model to use (see Details). Default is \code{lambda}.}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{phylolm}}
}
\value{
The function \code{clade_phylm} returns a list with the following
components:

\code{formula}: The formula

\code{full.model.estimates}: Coefficients, aic and the optimised
value of the phylogenetic parameter (e.g. \code{lambda}) for the full model
without deleted species.

\code{sensi.estimates}: A data frame with all simulation
estimates. Each row represents a deleted clade for a tree iteration. Columns report the calculated
regression intercept (\code{intercept}), difference between simulation
intercept and full model intercept (\code{DIFintercept}), the percentage of change
in intercept compared to the full model (\code{intercept.perc}) and intercept
p-value (\code{pval.intercept}). All these parameters are also reported for the regression
slope (\code{DIFestimate} etc.). Additionally, model aic value (\code{AIC}) and
the optimised value (\code{optpar}) of the phylogenetic parameter 
(e.g. \code{kappa} or \code{lambda}, depending on the phylogenetic model used) 
are reported.

\code{null.dist}: A data frame with estimates for the null distributions
for all clades analysed.

\code{data}: Original full dataset.

\code{errors}: Clades and/or trees where deletion resulted in errors.
}
\description{
Estimate the impact on model estimates of phylogenetic linear regression after 
removing clades from the analysis and evaluating uncertainty in trees topology.
}
\details{
This function sequentially removes one clade at a time, fits a phylogenetic
linear regression model using \code{\link[phylolm]{phylolm}} and stores the
results. The impact of of a specific clade on model estimates is calculated by the
comparison between the full model (with all species) and the model without 
the species belonging to a clade. It repeats this operation using n trees, 
randomly picked in a multiPhylo file.

 Additionally, to account for the influence of the number of species on each 
 clade (clade sample size), this function also estimate a null distribution of slopes
 expected for the number of species in a given clade. This is done by fitting
 models without the same number of species in the given clade. 
 The number of simulations to be performed is set by 'n.sim'. To test if the 
 clade influence differs from the null expectation for a clade of that size, 
 a randomization test can be performed using 'summary(x)'. 

All phylogenetic models from \code{phylolm} can be used, i.e. \code{BM},
\code{OUfixedRoot}, \code{OUrandomRoot}, \code{lambda}, \code{kappa},
\code{delta}, \code{EB} and \code{trend}. See ?\code{phylolm} for details.

\code{clade_phylm} detects influential clades based on
difference in intercept and/or slope when removing a given clade compared
to the full model including all species. This is done for n trees in the multiphylo file.

Currently, this function can only implement simple linear models (i.e. 
\eqn{y = a + bx}). In the future we will implement more complex models.

Output can be visualised using \code{sensi_plot}.
}
\examples{
\dontrun{
# Load data:
data(primates)
# run analysis:
clade_tree <- tree_clade_phylm(log(sexMaturity) ~ log(adultMass), 
phy = primates$phy, data = primates$data, clade.col = "family", n.sim = 50, n.tree = 5)
# To check summary results and most influential clades:
summary(clade_tree)
# Visual diagnostics for clade removal:
sensi_plot(clade_tree)
# Specify which clade removal to plot:
sensi_plot(clade_tree)
sensi_plot(clade_tree, "Cercopithecidae")
sensi_plot(clade_tree, clade = "Cebidae", graphs = 2)
}
\dontshow{
data(primates)
# run analysis:
clade_tree <- tree_clade_phylm(log(sexMaturity) ~ log(adultMass), 
                              phy = primates$phy, data = primates$data, 
                              clade.col = "family", n.sim = 10, n.tree = 2)
# To check summary results and most influential clades:
summary(clade_tree)
}
}
\references{
Ho, L. S. T. and Ane, C. 2014. "A linear-time algorithm for 
Gaussian and non-Gaussian trait evolution models". Systematic Biology 63(3):397-408.
}
\seealso{
\code{\link[phylolm]{phylolm}}, \code{\link[sensiPhy]{tree_phylm}},
 \code{\link{clade_phylm}}, \code{\link{tree_clade_phyglm}},
\code{\link{sensi_plot}}
}
\author{
Gustavo Paterno, Caterina Penone & Gijsbert D.A. Werner
}
