% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TargetingModels.R
\name{plotTune}
\alias{plotTune}
\title{Visualize parameter tuning for minNumMutations and minNumSeqMutations}
\usage{
plotTune(tuneMtx, thresh, criterion = c("5mer", "3mer", "1mer", "3mer+1mer",
  "measured", "inferred"), pchs = 1, ltys = 2, cols = 1,
  plotLegend = TRUE, legendPos = "topright", legendHoriz = FALSE,
  legendCex = 1)
}
\arguments{
\item{tuneMtx}{a \code{matrix} or a \code{list} of matrices produced by either 
\link{minNumMutationsTune} or \link{minNumSeqMutationsTune}.
In the case of a list, it is assumed that each matrix corresponds
to a sample and that all matrices in the list were produced using
the same set of trial values of \code{minNumMutations} or 
\code{minNumSeqMutations}.}

\item{thresh}{a number or a vector of indicating the value or the range of values
of \code{minNumMutations} or \code{minNumSeqMutations} to plot. 
Should correspond to the columns of \code{tuneMtx}.}

\item{criterion}{one of \code{"5mer"}, \code{"3mer"}, \code{"1mer"}, or \code{"3mer+1mer"} 
(for \code{tuneMtx} produced by \link{minNumMutationsTune}), or either 
\code{"measured"} or \code{"inferred"} (for \code{tuneMtx} produced by 
\link{minNumSeqMutationsTune}).}

\item{pchs}{point types to pass on to \link{plot}.}

\item{ltys}{line types to pass on to \link{plot}.}

\item{cols}{colors to pass on to \link{plot}.}

\item{plotLegend}{whether to plot legend. Default is \code{TRUE}. Only applicable 
if \code{tuneMtx} is a named list with names of the matrices 
corresponding to the names of the samples.}

\item{legendPos}{position of legend to pass on to \link{legend}. Can be either a
numeric vector specifying x-y coordinates, or one of 
\code{"topright"}, \code{"center"}, etc. Default is \code{"topright"}.}

\item{legendHoriz}{whether to make legend horizontal. Default is \code{FALSE}.}

\item{legendCex}{numeric values by which legend should be magnified relative to 1.}
}
\description{
Visualize results from \link{minNumMutationsTune} and \link{minNumSeqMutationsTune}
}
\details{
For \code{tuneMtx} produced by \link{minNumMutationsTune}, for each sample, depending on
          \code{criterion}, the numbers of 5-mers for which substitution rates are directly computed
          (\code{"5mer"}), inferred based on inner 3-mers (\code{"3mer"}), inferred based on 
          central 1-mers (\code{"1mer"}), or inferred based on inner 3-mers and central 1-mers
          (\code{"3mer+1mer"}) are plotted on the y-axis against values of \code{minNumMutations} 
          on the x-axis.

          For \code{tuneMtx} produced by \link{minNumSeqMutationsTune}, for each sample, depending on
          \code{criterion}, the numbers of 5-mers for which mutability rates are directly measured
          (\code{"measured"}) or inferred (\code{"inferred"}) are plotted on the y-axis against values
          of \code{minNumSeqMutations} on the x-axis.
          
          Note that legends will be plotted only if \code{tuneMtx} is a supplied as a named \code{list}
          of matrices, ideally with names of each \code{matrix} corresponding to those of the samples 
          based on which the matrices were produced, even if \code{plotLegend=TRUE}.
}
\examples{
\donttest{
# Subset example data to one isotype and sample as demos
data(ExampleDb, package="alakazam")
db <- subset(ExampleDb, ISOTYPE == "IgA")

tuneMtx = list()
for (i in 1:length(unique(db$SAMPLE))) {
    # Get data corresponding to current sample
    curDb = db[db$SAMPLE==unique(db$SAMPLE)[i], ]
    
    # Count the number of mutations per 5-mer
    subCount = createSubstitutionMatrix(db=curDb, model="S", multipleMutation="independent",
                                        returnModel="5mer", numMutationsOnly=TRUE)
    
    # Tune over minNumMutations = 5..50
    subTune = minNumMutationsTune(subCount, seq(from=5, to=50, by=5))
    
    tuneMtx = c(tuneMtx, list(subTune))
}

# Name tuneMtx after sample names 
names(tuneMtx) = unique(db$SAMPLE)

# plot with legend for both samples for a subset of minNumMutations values
plotTune(tuneMtx, thresh=c(5, 15, 25, 40), criterion="3mer",
         pchs=16:17, ltys=1:2, cols=2:3, 
         plotLegend=TRUE, legendPos=c(5, 100))

# plot for only 1 sample for all the minNumMutations values (no legend)
plotTune(tuneMtx[[1]], thresh=seq(from=5, to=50, by=5), criterion="3mer")
}

}
\seealso{
See \link{minNumMutationsTune} and \link{minNumSeqMutationsTune} for generating 
          \code{tuneMtx}.
}
