% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bind-event.R
\name{bindEvent}
\alias{bindEvent}
\title{Make an object respond only to specified reactive events}
\usage{
bindEvent(
  x,
  ...,
  ignoreNULL = TRUE,
  ignoreInit = FALSE,
  once = FALSE,
  label = NULL
)
}
\arguments{
\item{x}{An object to wrap so that is triggered only when a the specified
event occurs.}

\item{...}{One or more expressions that represents the event; this can be a
simple reactive value like \code{input$click}, a call to a reactive expression
like \code{dataset()}, or even a complex expression inside curly braces. If
there are multiple expressions in the \code{...}, then it will take a dependency
on all of them.}

\item{ignoreNULL}{Whether the action should be triggered (or value
calculated) when the input is \code{NULL}. See Details.}

\item{ignoreInit}{If \code{TRUE}, then, when the eventified object is first
created/initialized, don't trigger the action or (compute the value). The
default is \code{FALSE}. See Details.}

\item{once}{Used only for observers. Whether this \code{observer} should be
immediately destroyed after the first time that the code in the observer is
run. This pattern is useful when you want to subscribe to a event that
should only happen once.}

\item{label}{A label for the observer or reactive, useful for debugging.}
}
\description{
Modify an object to respond to "event-like" reactive inputs, values, and
expressions. \code{bindEvent()} can be used with reactive expressions, render
functions, and observers. The resulting object takes a reactive dependency on
the \code{...} arguments, and not on the original object's code. This can, for
example, be used to make an observer execute only when a button is pressed.

\code{bindEvent()} was added in Shiny 1.6.0. When it is used with \code{\link[=reactive]{reactive()}} and
\code{\link[=observe]{observe()}}, it does the same thing as \code{\link[=eventReactive]{eventReactive()}} and
\code{\link[=observeEvent]{observeEvent()}}. However, \code{bindEvent()} is more flexible: it can be combined
with \code{\link[=bindCache]{bindCache()}}, and it can also be used with \code{render} functions (like
\code{\link[=renderText]{renderText()}} and \code{\link[=renderPlot]{renderPlot()}}).
}
\section{Details}{


Shiny's reactive programming framework is primarily designed for calculated
values (reactive expressions) and side-effect-causing actions (observers)
that respond to \emph{any} of their inputs changing. That's often what is
desired in Shiny apps, but not always: sometimes you want to wait for a
specific action to be taken from the user, like clicking an
\code{\link[=actionButton]{actionButton()}}, before calculating an expression or taking an action. A
reactive value or expression that is used to trigger other calculations in
this way is called an \emph{event}.

These situations demand a more imperative, "event handling" style of
programming that is possible--but not particularly intuitive--using the
reactive programming primitives \code{\link[=observe]{observe()}} and \code{\link[=isolate]{isolate()}}. \code{bindEvent()}
provides a straightforward API for event handling that wraps \code{observe} and
\code{isolate}.

The \code{...} arguments are captured as expressions and combined into an
\strong{event expression}. When this event expression is invalidated (when its
upstream reactive inputs change), that is an \strong{event}, and it will cause
the original object's code to execute.

Use \code{bindEvent()} with \code{observe()} whenever you want to \emph{perform an action}
in response to an event. (This does the same thing as \code{\link[=observeEvent]{observeEvent()}},
which was available in Shiny prior to version 1.6.0.) Note that
"recalculate a value" does not generally count as performing an action --
use \code{\link[=reactive]{reactive()}} for that.

Use \code{bindEvent()} with \code{reactive()} to create a \emph{calculated value} that
only updates in response to an event. This is just like a normal \link[=reactive]{reactive expression} except it ignores all the usual invalidations that
come from its reactive dependencies; it only invalidates in response to the
given event. (This does the same thing as \code{\link[=eventReactive]{eventReactive()}}, which was
available in Shiny prior to version 1.6.0.)

\code{bindEvent()} is often used with \code{\link[=bindCache]{bindCache()}}.
}

\section{ignoreNULL and ignoreInit}{


\code{bindEvent()} takes an \code{ignoreNULL} parameter that affects behavior when
the event expression evaluates to \code{NULL} (or in the special case of an
\code{\link[=actionButton]{actionButton()}}, \code{0}). In these cases, if \code{ignoreNULL} is \code{TRUE}, then it
will raise a silent \link[=validate]{validation} error. This is useful behavior
if you don't want to do the action or calculation when your app first
starts, but wait for the user to initiate the action first (like a "Submit"
button); whereas \code{ignoreNULL=FALSE} is desirable if you want to initially
perform the action/calculation and just let the user re-initiate it (like a
"Recalculate" button).

\code{bindEvent()} also takes an \code{ignoreInit} argument. By default, reactive
expressions and observers will run on the first reactive flush after they
are created (except if, at that moment, the event expression evaluates to
\code{NULL} and \code{ignoreNULL} is \code{TRUE}). But when responding to a click of an
action button, it may often be useful to set \code{ignoreInit} to \code{TRUE}. For
example, if you're setting up an observer to respond to a dynamically
created button, then \code{ignoreInit = TRUE} will guarantee that the action
will only be triggered when the button is actually clicked, instead of also
being triggered when it is created/initialized. Similarly, if you're
setting up a reactive that responds to a dynamically created button used to
refresh some data (which is then returned by that \code{reactive}), then you
should use \code{reactive(...) \%>\% bindEvent(..., ignoreInit = TRUE)} if you
want to let the user decide if/when they want to refresh the data (since,
depending on the app, this may be a computationally expensive operation).

Even though \code{ignoreNULL} and \code{ignoreInit} can be used for similar purposes
they are independent from one another. Here's the result of combining
these:

\describe{
\item{\code{ignoreNULL = TRUE} and \code{ignoreInit = FALSE}}{
This is the default. This combination means that reactive/observer code
will run every time that event expression is not
\code{NULL}. If, at the time of creation, the event expression happens
to \emph{not} be \code{NULL}, then the code runs.
}
\item{\code{ignoreNULL = FALSE} and \code{ignoreInit = FALSE}}{
This combination means that reactive/observer code will
run every time no matter what.
}
\item{\code{ignoreNULL = FALSE} and \code{ignoreInit = TRUE}}{
This combination means that reactive/observer code will
\emph{not} run at the time of creation (because \code{ignoreInit = TRUE}),
but it will run every other time.
}
\item{\code{ignoreNULL = TRUE} and \code{ignoreInit = TRUE}}{
This combination means that reactive/observer code will
\emph{not} at the time of creation (because  \code{ignoreInit = TRUE}).
After that, the reactive/observer code will run every time that
the event expression is not \code{NULL}.
}
}
}

\section{Types of objects}{


\code{bindEvent()} can be used with reactive expressions, observers, and shiny
render functions.

When \code{bindEvent()} is used with \code{reactive()}, it creates a new reactive
expression object.

When \code{bindEvent()} is used with \code{observe()}, it alters the observer in
place. It can only be used with observers which have not yet executed.
}

\section{Combining events and caching}{


In many cases, it makes sense to use \code{bindEvent()} along with
\code{bindCache()}, because they each can reduce the amount of work done on the
server. For example, you could have \link{sliderInput}s \code{x} and \code{y} and a
\code{reactive()} that performs a time-consuming operation with those values.
Using \code{bindCache()} can speed things up, especially if there are multiple
users. But it might make sense to also not do the computation until the
user sets both \code{x} and \code{y}, and then clicks on an \link{actionButton} named
\code{go}.

To use both caching and events, the object should first be passed to
\code{bindCache()}, then \code{bindEvent()}. For example:\if{html}{\out{<div class="sourceCode NA">}}\preformatted{r <- reactive(\{
   Sys.sleep(2)  # Pretend this is an expensive computation
   input$x * input$y
 \}) \%>\%
 bindCache(input$x, input$y) \%>\%
 bindEvent(input$go)
}\if{html}{\out{</div>}}

Anything that consumes \code{r()} will take a reactive dependency on the event
expression given to \code{bindEvent()}, and not the cache key expression given to
\code{bindCache()}. In this case, it is just \code{input$go}.
}

