% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simmr_mcmc_tdf.R
\name{simmr_mcmc_tdf}
\alias{simmr_mcmc_tdf}
\title{Estimate correction factors from stable isotope data with known dietary
proportions}
\usage{
simmr_mcmc_tdf(
  simmr_in,
  p = matrix(rep(1/simmr_in$n_sources, simmr_in$n_sources), ncol = simmr_in$n_sources,
    nrow = simmr_in$n_obs, byrow = TRUE),
  prior_control = list(c_mean_est = rep(2, simmr_in$n_tracers), c_sd_est = rep(2,
    simmr_in$n_tracers)),
  mcmc_control = list(iter = 10000, burn = 1000, thin = 10, n.chain = 4)
)
}
\arguments{
\item{simmr_in}{An object created via the function \code{\link{simmr_load}}}

\item{p}{The known dietary proportions for the feeding study. Dietary proportions should be given per individual (even if they are all identical)}

\item{prior_control}{A list of values including arguments named \code{means}
and \code{sd} which represent the prior means and standard deviations of the
correction factors. These can usually be left at their default values unless 
you wish to include to include prior information on them.}

\item{mcmc_control}{A list of values including arguments named \code{iter}
(number of iterations), \code{burn} (size of burn-in), \code{thin} (amount
of thinning), and \code{n.chain} (number of MCMC chains).}
}
\value{
An object of class \code{simmr_tdf} with two named top-level
components: 
\item{input}{The \code{simmr_input} object given to the
\code{simmr_mcmc} function} 
\item{output}{A set of MCMC chains of class
\code{mcmc.list} from the coda package. These can be analysed using 
\code{\link{summary.simmr_output_tdf}}}.
}
\description{
This function runs a slightly different version of the main 
\code{\link{simmr_mcmc}} function with the key difference that it estimates
the correction factors (sometimes called trophic enrichment or trophic
discrimination factors; TEFs/TDFs) for a given set of dietary proportions.
}
\details{
The idea is that this code can be used for feeding studies where an 
organism is fed a known proportional diet with a view to estimating
the correction factors to be used in a later stable isotope mixing 
model when the organisms are observed in the field. 

The main argument of the function is an object created from 
\code{\link{simmr_load}} which contains mixture data on a number of tracers
and food source means and standard deviations. Any correction factors 
included in this object will be ignored. The known dietary proportions should be provided for each individual (i.e. should be a matrix with the same number of rows as \code{mix}). It is advisable to have multiple different dietary proportion values as part of the feeding experimental design

The output of the function is a posterior distribution on the correction
factors for each food source. Just like the output from
\code{\link{simmr_mcmc}}, this should be checked for convergence. Examples
are included below to help assist with this check and further plots

If, after running \code{\link{simmr_mcmc_tdf}} the convergence diagnostics in
\code{\link{summary.simmr_output_tdf}} are not satisfactory, the values of
\code{iter}, \code{burn} and \code{thin} in \code{mcmc_control} should be
increased by e.g. a factor of 10.
}
\examples{
\dontrun{
## Example of estimating TDFs for a simple system with known dietary proportions

# Data set 1: 10 obs on 2 isos, 4 sources, with tefs and concdep
# Assume p = c(0.25, 0.25, 0.25, 0.25)

# The data
mix = matrix(c(-10.13, -10.72, -11.39, -11.18, -10.81, -10.7, -10.54, 
-10.48, -9.93, -9.37, 11.59, 11.01, 10.59, 10.97, 11.52, 11.89, 
11.73, 10.89, 11.05, 12.3), ncol=2, nrow=10)
colnames(mix) = c('d13C','d15N')
s_names=c('Source A','Source B','Source C','Source D')
s_means = matrix(c(-14, -15.1, -11.03, -14.44, 3.06, 7.05, 13.72, 5.96), ncol=2, nrow=4)
s_sds = matrix(c(0.48, 0.38, 0.48, 0.43, 0.46, 0.39, 0.42, 0.48), ncol=2, nrow=4)
conc = matrix(c(0.02, 0.1, 0.12, 0.04, 0.02, 0.1, 0.09, 0.05), ncol=2, nrow=4)

# Load into simmr
simmr_tdf = simmr_load(mixtures=mix,
                     source_names=s_names,
                     source_means=s_means,
                     source_sds=s_sds,
                     concentration_means = conc)

# Plot
plot(simmr_tdf)

# MCMC run
simmr_tdf_out = simmr_mcmc_tdf(simmr_tdf, 
p = matrix(rep(1/simmr_tdf$n_sources, 
simmr_tdf$n_sources),
ncol = simmr_tdf$n_sources, 
nrow = simmr_tdf$n_obs, byrow = TRUE))

# Summary
summary(simmr_tdf_out,type='diagnostics')
summary(simmr_tdf_out,type='quantiles')

# Now put these corrections back into the model and check the 
# iso-space plots and dietary output
simmr_tdf_2 = simmr_load(mixtures=mix,
                     source_names=s_names,
                     source_means=s_means,
                     source_sds=s_sds,
                     correction_means = simmr_tdf_out$c_mean_est,
                     correction_sds = simmr_tdf_out$c_sd_est,
                     concentration_means = conc)

# Plot with corrections now
plot(simmr_tdf_2)

simmr_tdf_2_out = simmr_mcmc(simmr_tdf_2)
summary(simmr_tdf_2_out, type = 'diagnostics')
plot(simmr_tdf_2_out, type = 'boxplot')
}

}
\references{
Andrew C. Parnell, Donald L. Phillips, Stuart Bearhop, Brice X.
Semmens, Eric J. Ward, Jonathan W. Moore, Andrew L. Jackson, Jonathan Grey,
David J. Kelly, and Richard Inger. Bayesian stable isotope mixing models.
Environmetrics, 24(6):387–399, 2013.

Andrew C Parnell, Richard Inger, Stuart Bearhop, and Andrew L Jackson.
Source partitioning using stable isotopes: coping with too much variation.
PLoS ONE, 5(3):5, 2010.
}
\seealso{
\code{\link{simmr_load}} for creating objects suitable for this
function, \code{\link{simmr_mcmc}} for estimating dietary proportions, 
\code{\link{plot.simmr_input}} for creating isospace plots,
\code{\link{summary.simmr_output_tdf}} for summarising output
}
\author{
Andrew Parnell <andrew.parnell@mu.ie>
}
