% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pseudo_r2.R
\name{r2}
\alias{r2}
\title{Compute r-squared of (generalized) linear (mixed) models}
\usage{
r2(x, n = NULL)
}
\arguments{
\item{x}{Fitted model of class \code{lm}, \code{glm}, \code{lmerMod}/\code{lme}
or \code{glmerMod}.}

\item{n}{Optional, a \code{lmerMod} object, representing the fitted null-model
(unconditional model) to \code{x}. If \code{n} is given, the pseudo-r-squared
for random intercept and random slope variances are computed
(\cite{Kwok et al. 2008}) as well as the Omega squared value
(\cite{Xu 2003}). See 'Examples' and 'Details'.}
}
\value{
\itemize{
          \item For linear models, the r-squared and adjusted r-squared values.
          \item For linear mixed models, the r-squared and Omega-squared values.
          \item For \code{glm} objects, Cox & Snell's and Nagelkerke's pseudo r-squared values.
          \item For \code{glmerMod} objects, Tjur's coefficient of determination.
        }
}
\description{
Compute R-squared values of linear (mixed) models, or
               pseudo-R-squared values for generalized linear (mixed) models.
}
\details{
For linear models, the r-squared and adjusted r-squared value is returned,
         as provided by the \code{summary}-function.
         \cr \cr
         For linear mixed models, an r-squared approximation by computing the
         correlation between the fitted and observed values, as suggested by
         \cite{Byrnes (2008)}, is returned as well as a simplified version of
         the Omega-squared value (1 - (residual variance / response variance),
         \cite{Xu (2003)}, \cite{Nakagawa, Schielzeth 2013}), unless \code{n}
         is specified.
         \cr \cr
         If \code{n} is given, for linear mixed models pseudo r-squared measures based
         on the variances of random intercept (tau 00, between-group-variance)
         and random slope (tau 11, random-slope-variance), as well as the
         r-squared statistics as proposed by \cite{Snijders and Bosker 2012} and
         the Omega-squared value (1 - (residual variance full model / residual
         variance null model)) as suggested by \cite{Xu (2003)} are returned.
         \cr \cr
         For generalized linear models, Cox & Snell's and Nagelkerke's
         pseudo r-squared values are returned.
         \cr \cr
         For generalized linear mixed models, the coefficient of determination
         as suggested by \cite{Tjur (2009)} (see also \code{\link{cod}}). Note
         that \emph{Tjur's D} is restricted to models with binary response.
         \cr \cr
         More ways to compute coefficients of determination are shown
         in this great \href{https://github.com/bbolker/mixedmodels-misc/blob/master/glmmFAQ.rmd#model-summaries-goodness-of-fit-decomposition-of-variance-etc}{GLMM faq}.
         Furthermore, see \code{\link[MuMIn]{r.squaredGLMM}} or
         \code{\link[piecewiseSEM]{rsquared}} for conditional and marginal
         r-squared values for GLMM's.
}
\note{
If \code{n} is given, the Pseudo-R2 statistic is the proportion of
         explained variance in the random effect after adding co-variates or
         predictors to the model, or in short: the proportion of the explained
         variance in the random effect of the full (conditional) model \code{x}
         compared to the null (unconditional) model \code{n}.
         \cr \cr
         The Omega-squared statistics, if \code{n} is given, is 1 - the proportion
         of the residual variance of the full model compared to the null model's
         residual variance, or in short: the the proportion of the residual
         variation explained by the covariates.
         \cr \cr
         The r-squared statistics for linear mixed models, if the unconditional
         model is also specified (see \code{n}), is the difference of the total
         variance of the null and full model divided by the total variance of
         the null model.
         \cr \cr
         Alternative ways to assess the "goodness-of-fit" is to compare the ICC
         of the null model with the ICC of the full model (see \code{\link{icc}}).
}
\examples{
library(sjmisc)
library(lme4)
fit <- lmer(Reaction ~ Days + (Days | Subject), sleepstudy)
r2(fit)

data(efc)
fit <- lm(barthtot ~ c160age + c12hour, data = efc)
r2(fit)

# Pseudo-R-squared values
efc$services <- ifelse(efc$tot_sc_e > 0, 1, 0)
fit <- glm(services ~ neg_c_7 + c161sex + e42dep,
           data = efc, family = binomial(link = "logit"))
r2(fit)

# Pseudo-R-squared values for random effect variances
fit <- lmer(Reaction ~ Days + (Days | Subject), sleepstudy)
fit.null <- lmer(Reaction ~ 1 + (Days | Subject), sleepstudy)
r2(fit, fit.null)


}
\references{
\itemize{
              \item \href{http://glmm.wikidot.com/faq}{DRAFT r-sig-mixed-models FAQ}
              \item Byrnes, J. 2008. Re: Coefficient of determination (R^2) when using lme() (\url{https://stat.ethz.ch/pipermail/r-sig-mixed-models/2008q2/000713.html})
              \item Kwok OM, Underhill AT, Berry JW, Luo W, Elliott TR, Yoon M. 2008. Analyzing Longitudinal Data with Multilevel Models: An Example with Individuals Living with Lower Extremity Intra-Articular Fractures. Rehabilitation Psychology 53(3): 370–86. \doi{10.1037/a0012765}
              \item Nakagawa S, Schielzeth H. 2013. A general and simple method for obtaining R2 from generalized linear mixed-effects models. Methods in Ecology and Evolution, 4(2):133–142. \doi{10.1111/j.2041-210x.2012.00261.x}
              \item Rabe-Hesketh S, Skrondal A. 2012. Multilevel and longitudinal modeling using Stata. 3rd ed. College Station, Tex: Stata Press Publication
              \item Raudenbush SW, Bryk AS. 2002. Hierarchical linear models: applications and data analysis methods. 2nd ed. Thousand Oaks: Sage Publications
              \item Snijders TAB, Bosker RJ. 2012. Multilevel analysis: an introduction to basic and advanced multilevel modeling. 2nd ed. Los Angeles: Sage
              \item Xu, R. 2003. Measuring explained variation in linear mixed effects models. Statist. Med. 22:3527-3541. \doi{10.1002/sim.1572}
              \item Tjur T. 2009. Coefficients of determination in logistic regression models - a new proposal: The coefficient of discrimination. The American Statistician, 63(4): 366-372
            }
}
\seealso{
\code{\link{rmse}} for more methods to assess model quality.
}
