% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/amplitude.R
\name{addAM}
\alias{addAM}
\title{Add amplitude modulation}
\usage{
addAM(
  x,
  samplingRate = NULL,
  amDep = 25,
  amFreq = 30,
  amType = c("logistic", "sine")[1],
  amShape = 0,
  invalidArgAction = c("adjust", "abort", "ignore")[1],
  plot = FALSE,
  play = FALSE,
  saveAudio = NULL,
  reportEvery = NULL,
  cores = 1
)
}
\arguments{
\item{x}{path to a folder, one or more wav or mp3 files c('file1.wav',
'file2.mp3'), Wave object, numeric vector, or a list of Wave objects or
numeric vectors}

\item{samplingRate}{sampling rate of \code{x} (only needed if \code{x} is a
numeric vector)}

\item{amDep}{amplitude modulation (AM) depth, \%. 0: no change; 100: AM with
amplitude range equal to the dynamic range of the sound (anchor format)}

\item{amFreq}{AM frequency, Hz (anchor format)}

\item{amType}{"sine" = sinusoidal, "logistic" = logistic (default)}

\item{amShape}{ignore if amType = "sine", otherwise determines the shape of
non-sinusoidal AM: 0 = ~sine, -1 = notches, +1 = clicks (anchor format)}

\item{invalidArgAction}{what to do if an argument is invalid or outside the
range in \code{permittedValues}: 'adjust' = reset to default value, 'abort'
= stop execution, 'ignore' = throw a warning and continue (may crash)}

\item{plot}{if TRUE, plots the amplitude modulation}

\item{play}{if TRUE, plays the processed audio}

\item{saveAudio}{full (!) path to folder for saving the processed audio; NULL
= don't save, '' = same as input folder (NB: overwrites the originals!)}

\item{reportEvery}{when processing multiple inputs, report estimated time
left every ... iterations (NULL = default, NA = don't report)}

\item{cores}{number of cores for parallel processing}
}
\description{
Adds sinusoidal or logistic amplitude modulation to a sound. This produces
additional harmonics in the spectrum at ±am_freq around each original
harmonic and makes the sound rough. The optimal frequency for creating a
perception of roughness is ~70 Hz (Fastl & Zwicker "Psychoacoustics").
Sinusoidal AM creates a single pair of new harmonics, while non-sinusoidal AM
creates more extra harmonics (see examples).
}
\examples{
sound1 = soundgen(pitch = c(200, 300), addSilence = 0)
s1 = addAM(sound1, 16000, amDep = c(0, 50, 0), amFreq = 75, plot = TRUE)
# playme(s1)
\dontrun{
# Parameters can be specified as in the soundgen() function, eg:
s2 = addAM(sound1, 16000,
         amDep = list(time = c(0, 50, 52, 200, 201, 300),
                      value = c(0, 0, 35, 25, 0, 0)),
         plot = TRUE, play = TRUE)

# Sinusoidal AM produces exactly 2 extra harmonics at ±am_freq
# around each f0 harmonic:
s3 = addAM(sound1, 16000, amDep = 30, amFreq = c(50, 80),
           amType = 'sine', plot = TRUE, play = TRUE)
spectrogram(s3, 16000, windowLength = 150, ylim = c(0, 2))

# Non-sinusoidal AM produces multiple new harmonics,
# which can resemble subharmonics...
s4 = addAM(sound1, 16000, amDep = 70, amFreq = 50, amShape = -1,
           plot = TRUE, play = TRUE)
spectrogram(s4, 16000, windowLength = 150, ylim = c(0, 2))

# ...but more often look like sidebands
sound3 = soundgen(sylLen = 600, pitch = c(800, 1300, 1100), addSilence = 0)
s5 = addAM(sound3, 16000, amDep = c(0, 30, 100, 40, 0),
           amFreq = 105, amShape = -.3,
           plot = TRUE, play = TRUE)
spectrogram(s5, 16000, ylim = c(0, 5))

# Feel free to add AM stochastically:
s6 = addAM(sound1, 16000,
           amDep = rnorm(10, 40, 20), amFreq = rnorm(20, 70, 20),
           plot = TRUE, play = TRUE)
spectrogram(s6, 16000, windowLength = 150, ylim = c(0, 2))

# If am_freq is locked to an integer ratio of f0, we can get subharmonics
# For ex., here is with pitch 400-600-400 Hz (soundgen interpolates pitch
# on a log scale and am_freq on a linear scale, so we align them by extracting
# a long contour on a log scale for both)
con = getSmoothContour(anchors = c(400, 600, 400),
                       len = 20, thisIsPitch = TRUE)
s = soundgen(sylLen = 1500, pitch = con, amFreq = con/3, amDep = 30,
             plot = TRUE, play = TRUE, ylim = c(0, 3))

# Process all files in a folder and save the modified audio
addAM('~/Downloads/temp', saveAudio = '~/Downloads/temp/AM',
      amFreq = 70, amDep = c(0, 50))
}
}
