% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/GuanTestUnif.R
\name{GuanTestUnif}
\alias{GuanTestUnif}
\title{Nonparametric Test of Isotropy Using the Sample Semivariogram}
\usage{
GuanTestUnif(spdata, lagmat, A, df, subblock.dims, h = 1, kernel = "norm",
  truncation = 1.5, xlims, ylims, grid.spacing = 1, subblock.h,
  sig.est.finite = T)
}
\arguments{
\item{spdata}{An \eqn{n} by \eqn{3} matrix. The first two columns provide \eqn{(x,y)} spatial coordinates. The third column provides data values at the coordinates.}

\item{lagmat}{A \eqn{k} by \eqn{2} matrix of spatial lags. Each row corresponds to a lag of the form \eqn{(x.lag, y.lag)} for which the semivariogram value will be estimated.}

\item{A}{A \eqn{d} by \eqn{k} contrast matrix. The contrasts correspond to contrasts of the estimated semivariogram at the lags given in 'lagmat'.}

\item{df}{A scalar indicating the row rank of A. This value gives the degrees of freedom for the asymptotic Chi-sq distribution used to compute the p-value.}

\item{subblock.dims}{A vector of length two corresponding to the width and height of the moving windows used to estimate the asymptotic variance-covariance matrix. If window width does not evenly divide the width of the sampling region, some data will be ommited during subsampling, i.e., function does not handle partial windows. Same applies to window height and height of sampling region.}

\item{h}{A scalar giving the bandwidth for the kernel smoother. The same bandwidth is used for lags in both the x and y directions.}

\item{kernel}{A string taking one of the following values: "norm", "ep", "cos", or "unif", for the normal, Epanechnikov, cosine, or uniform kernel functions. Defaults to normal.}

\item{truncation}{A scalar providing the truncation value for the normal density if 'kernel' is given as "norm".}

\item{xlims}{A vector of length two providing the lower and upper x-limits of the sampling region.}

\item{ylims}{A vector of length two providing the lower and upper y-limits of the sampling region.}

\item{grid.spacing}{A scalar indicating the spacing of the underlying grid laid on the sampling region to create moving windows. The same spacing is used in the x and y directions.}

\item{subblock.h}{A scalar giving the bandwidth used for the kernel smoother when estimating the semivariogram on the moving windows (blocks). It is recommended to be less than 1 to maintain nominal test size.}

\item{sig.est.finite}{Logical. True provides a finite sample correction in estimating Sigma (see Guan et. al. (2004) Section 4.2.2). False provides the empirical variance-covariance matrix of sample semivariogram values computed via the moving windows.}
}
\value{
\item{gamma.hat}{A matrix of the spatial lags provided and the semivariogram point estimates at those lags used to construct the test statistic.}
\item{sigma.hat}{The estimate of asymptotic variance-covariance matrix, Sigma, used to construct test statistic.}
\item{n.subblocks}{The number of moving windows (blocks) used to estimate Sigma.}
\item{test.stat}{The calculated test statistic.}
\item{pvalue.finite}{The approximate, finite-sample adjusted p-value computed by using the moving windows (see Guan et. al. (2004), Section 3.3 for details).}
\item{pvalue.chisq}{The p-value computed using the asymptotic Chi-sq distribution.}
\item{h.sb}{The value of h (bandwidth) used to estimate the semivariogram on the moving windows (blocks).}
}
\description{
This function performs the nonparametric test of isotropy from Guan et. al. (2004) for spatial data with uniformly distributed sampling locations. See Guan et. al. (2004) for more details.
}
\details{
This function currently only supports square and rectangular sampling regions and does not currently support partial blocks. For example, suppose the sampling region runs from 0 to 20 in the x-direction and from 0 to 30 in the y-direction and an underlying grid of 1 by 1 is laid over the sampling region. Then an ideal value of subblock.dims would be (2,3) since its entries evenly divide the width (20) and height (30), respectively, of the sampling region. Using the vector (3, 4.5) would imply that some data will not be used in the moving windows since these values would create partial blocks in the sampling region.

To preserve the spatial dependence structure, the spatial blocks should have the same shape (i.e. square or rectangle) and orientation as the entire sampling domain.
}
\examples{
library(mvtnorm)
set.seed(1)
#Sample Size
N <- 300
#Set parameter values for exponential covariance function
sigma.sq <- 1
tau.sq <- 0.0
phi <- 1/4
#Generate sampling locations
coords <-  cbind(runif(N,0,16), runif(N,0,16))
D <-  as.matrix(dist(coords))
R <- sigma.sq * exp(-phi*D)
R <- R + diag(tau.sq, nrow = N, ncol = N)
#Simulate Gaussian spatial data
z <- rmvnorm(1,rep(0,N), R, method = "chol")
z <- z - mean(z)
z <- t(z)
mydata <- cbind(coords, z)
mylags = rbind(c(1,0), c(0, 1), c(1, 1), c(-1,1))
myA = rbind(c(1, -1, 0 , 0), c(0, 0, 1, -1))
mysb.dims = c(4,4)
myh = 0.7
myh.sb = 0.8
my.xlims = c(0, 20)
my.ylims = c(0, 20)
tr <- GuanTestUnif(mydata, mylags, myA, df = 2, mysb.dims, myh, "norm", 1.5,
 my.xlims, my.ylims, 1, myh.sb)
tr

#Simulate data from anisotropic covariance function
aniso.angle <- pi/4
aniso.ratio <- 2
coordsA <- coords.aniso(coords, c(aniso.angle, aniso.ratio))
Da <- as.matrix(dist(coordsA))
R <- sigma.sq * exp(-phi*Da)
R <- R + diag(tau.sq, nrow = N, ncol = N)
z <- rmvnorm(1,rep(0,N), R, method = c("chol"))
z <-  z-mean(z)
z <- t(z)
mydata <- cbind(coords, z)
#Run the test on the data generated from an anisotropic covariance function
tr <- GuanTestUnif(mydata, mylags, myA, df = 2, mysb.dims, myh, "norm", 1.5,
my.xlims, my.ylims, 1, myh.sb)
tr
}
\references{
Guan, Y., Sherman, M., & Calvin, J. A. (2004). A nonparametric test for spatial isotropy using subsampling. \emph{Journal of the American Statistical Association}, 99(467), 810-821.
}
\seealso{
\code{\link{MaityTest}} \code{\link{GuanTestGrid}}
}
\keyword{external}

