% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/logistic.regression.R
\name{logistic.regression}
\alias{logistic.regression}
\title{Logistic and Auto-logistic regression}
\usage{
logistic.regression(ldata, y, x, penalty = TRUE, autologistic = FALSE,
  coords = NULL, bw = NULL, type = "inverse", style = "W",
  longlat = FALSE, ...)
}
\arguments{
\item{ldata}{data.frame object containing variables}

\item{y}{Dependent variable (y) in ldata}

\item{x}{Independent variable(s) (x) in ldata}

\item{penalty}{Apply regression penalty (TRUE/FALSE)}

\item{autologistic}{Add auto-logistic term (TRUE/FALSE)}

\item{coords}{Geographic coordinates for auto-logistic model matrix or sp object.}

\item{bw}{Distance bandwidth to calculate spatial lags (if empty neighbours result, need to increase bandwith). If not provided it will be calculated automatically based on the minimum distance that includes at least one neighbor.}

\item{type}{Neighbour weighting scheme (see autocov_dist)}

\item{style}{Type of neighbour matrix (Wij), default is mean of neighbours}

\item{longlat}{Are coordinates (coords) in geographic, lat/long (TRUE/FALSE)}

\item{...}{Additional arguments passed to lrm}
}
\value{
A list class object with the following components: 

    model lrm model object (rms class)

    bandwidth If AutoCov = TRUE returns the distance bandwidth used for the auto-covariance function

    diagTable data.frame of regression diagnostics

    coefTable data.frame of regression coefficients

    Residuals data.frame of residuals and standardized residuals

    AutoCov If an auto-logistic model, AutoCov represents lagged auto-covariance term

}
\description{
Performs a logistic (binomial) or auto-logistic (spatially lagged binomial) regression using maximum likelihood or penalized maximum likelihood estimation. 

}
\note{
It should be noted that the auto-logistic model (Besag 1972) is intended for exploratory analysis of spatial effects. Auto-logistic are know to underestimate the effect of environmental variables and tend to be unreliable (Dormann 2007).     

Wij matrix options under style argument - B is the basic binary coding, W is row standardised (sums over all links to n), C is globally standardised (sums over all links to n), U is equal to C divided by the number of neighbours (sums over all links to unity) and S is variance-stabilizing. 

Spatially lagged y defined as:  

  W(y)ij=sumj_(Wij yj)/ sumj_(Wij)

    where; Wij=1/Euclidean[i,j]


depends: rms, spdep

}
\examples{
require(sp)
require(spdep)
require(rms)                                                                       
data(meuse)
  coordinates(meuse) <- ~x+y  
    meuse@data <- data.frame(DepVar=rbinom(dim(meuse)[1], 1, 0.5), meuse@data)

#### Logistic model
lmodel <- logistic.regression(meuse@data, y='DepVar', x=c('dist','cadmium','copper')) 
  lmodel$model
    lmodel$diagTable
      lmodel$coefTable

### Auto-logistic model using 'autocov_dist' in 'spdep' package
lmodel <- logistic.regression(meuse@data, y='DepVar', x=c('dist','cadmium','copper'),
                              autologistic=TRUE, coords=coordinates(meuse), bw=5000) 
  lmodel$model
    lmodel$diagTable
      lmodel$coefTable
  est <- predict(lmodel$model, type='fitted.ind')

#### Add residuals, standardized residuals and estimated probabilities
VarNames <- rownames(lmodel$model$var)[-1]
  meuse@data$AutoCov <- lmodel$AutoCov
    meuse@data <- data.frame(meuse@data, Residual=lmodel$Residuals[,1], 
                             StdResid=lmodel$Residuals[,2], Probs=predict(lmodel$model, 
                             meuse@data[,VarNames],type='fitted') )  

#### Plot fit and probabilities
resid(lmodel$model, "partial", pl="loess") 
resid(lmodel$model, "partial", pl=TRUE)                  # plot residuals    
resid(lmodel$model, "gof")                               # global test of goodness of fit
lp1 <- resid(lmodel$model, "lp1")                        # Approx. leave-out linear predictors 
-2 * sum(meuse@data$DepVar * lp1 + log(1-plogis(lp1)))  # Approx leave-out-1 deviance
spplot(meuse, c('Probs'))                                # plot estimated probabilities at points

}
\author{
Jeffrey S. Evans  <jeffrey_evans@tnc.org>

}
\references{

Besag, J.E., (1972) Nearest-neighbour systems and the auto-logistic model for binary data. Journal of the Royal Statistical Society, Series B Methodological 34:75-83


Dormann, C.F., (2007) Assessing the validity of autologistic regression. Ecological Modelling 207:234-242   


Le Cessie, S., Van Houwelingen, J.C., (1992) Ridge estimators in logistic regression. Applied Statistics 41:191-201


Shao, J., (1993) Linear model selection by cross-validation. JASA 88:486-494

}

