% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_hawkes.r
\name{fit_hawkes}
\alias{fit_hawkes}
\alias{fit_hawkes_cbf}
\title{Self-exciting Hawkes process}
\usage{
fit_hawkes(
  times,
  parameters = list(),
  model = 1,
  marks = c(rep(1, length(times))),
  tmb_silent = TRUE,
  optim_silent = TRUE,
  ...
)

fit_hawkes_cbf(
  times,
  parameters = list(),
  model = 1,
  marks = c(rep(1, length(times))),
  background,
  background_integral,
  background_parameters,
  background_min,
  tmb_silent = TRUE,
  optim_silent = TRUE
)
}
\arguments{
\item{times}{A vector of numeric observed time points.}

\item{parameters}{A named list of parameter starting values:
\itemize{
\item \code{mu}, base rate of the Hawkes process (\eqn{\mu}),
\item \code{alpha} (supplied if \code{model = 1}), intensity jump after an event occurrence (\eqn{\alpha}),
\item \code{a_par} (supplied if \code{model} = 2), logit scale for \code{alpha} (must be 
set so that the intensity never goes negative and so that \code{alpha} <= \code{beta}), and
\item \code{beta}, exponential intensity decay (\eqn{\beta}).
}}

\item{model}{A numeric indicator specifying which model to fit:
\itemize{
\item \code{model = 1}, fits a Hawkes process with exponential decay (default);
\item \code{model = 2}, fits a Hawkes process with an \code{alpha} that can be negative.
}}

\item{marks}{Optional, a vector of numeric marks, defaults to 1 (i.e., no marks).}

\item{tmb_silent}{Logical, if \code{TRUE} (default) then
TMB inner optimisation tracing information will be printed.}

\item{optim_silent}{Logical, if \code{TRUE} (default) then for each iteration
\code{optim()} output will be printed.}

\item{...}{Additional arguments to pass to \code{optim()}}

\item{background}{A function taking one parameter and an
independent variable, returning a scalar.}

\item{background_integral}{The integral of \code{background}.}

\item{background_parameters}{The parameter(s)
for the background function \code{background}.
This could be a list of multiple values.}

\item{background_min}{A function taking one parameter and two points,
returns min of \code{background} between those points.}
}
\value{
A list containing components of the fitted model, see \code{TMB::MakeADFun}. Includes
\itemize{
\item \code{par}, a numeric vector of estimated parameter values;
\item \code{objective}, the objective function;
\item \code{gr}, the TMB calculated gradient function; and
\item \code{simulate}, a simulation function. 
}
}
\description{
Fit a Hawkes process using Template Model Builder (TMB). The function \code{fit_hawkes()} fits a
self-exciting Hawkes process with a constant background rate. Whereas, \code{fit_hawkes_cbf()} fits a Hawkes
processes with a user defined custom background function (non-homogeneous background rate).
}
\details{
A univariate Hawkes (Hawkes, AG. 1971) process is a self-exciting temporal point process
with conditional intensity function
\eqn{\lambda(t) = \mu + \alpha \Sigma_{i:\tau_i<t}e^{(-\beta * (t-\tau_i))}}. Here \eqn{\mu} is the
background rate of the process and the summation describes the historic temporal dependence.  Including mark information
results in the conditional intensity \eqn{\lambda(t; m(t)) = \mu + \alpha \Sigma_{i:\tau_i<t}m(\tau_i)e^{(-\beta * (t-\tau_i))}},
where \eqn{m(t)} is the temporal mark. This model can be fitted with \code{fit_hawkes()}.

An in-homogenous marked Hawkes process has conditional intensity function
\eqn{\lambda(t) = \mu(t) + \alpha \Sigma_{i:\tau_i<t}e^{(-\beta * (t-\tau_i))}}. Here, the
background rate, \eqn{\mu(t)}, varies in time. Such a model can be fitted
using \code{fit_hawkes_cbf()} where the parameters of the custom background function are estimated
before being passed to \code{TMB}.
}
\examples{
\donttest{
### ********************** ###
## A typical Hawkes model
### ********************** ###
data(retweets_niwa, package = "stelfi")
times <- unique(sort(as.numeric(difftime(retweets_niwa, min(retweets_niwa), units = "mins"))))
params <- c(mu = 0.05, alpha = 0.05, beta = 0.1)
fit <- fit_hawkes(times = times, parameters = params)
get_coefs(fit)
### ********************** ###
## A Hawkes model with marks (ETAS-type)
### ********************** ###
data("nz_earthquakes", package = "stelfi")
earthquakes <- nz_earthquakes[order(nz_earthquakes$origintime),]
earthquakes <- earthquakes[!duplicated(earthquakes$origintime), ]
times <- earthquakes$origintime
times <- as.numeric(difftime(times, min(times), units = "hours"))
marks <- earthquakes$magnitude
params <- c(mu = 0.05, alpha = 0.05, beta = 1)
fit <- fit_hawkes(times = times, parameters = params, marks = marks)
get_coefs(fit)
}
\donttest{
### ********************** ###
## A Hawkes process with a custom background function
### ********************** ###
if(require("hawkesbow")) {
times <- hawkesbow::hawkes(1000, fun = function(y) {1 + 0.5*sin(y)},
M = 1.5, repr = 0.5, family = "exp", rate = 2)$p
## The background function must take a single parameter and
## the time(s) at which it is evaluated
background <- function(params,times) {
A = exp(params[[1]])
B = stats::plogis(params[[2]]) * A
return(A + B  *sin(times))
}
## The background_integral function must take a
## single parameter and the time at which it is evaluated
background_integral <- function(params,x) {
        A = exp(params[[1]])
        B = stats::plogis(params[[2]]) * A
        return((A*x)-B*cos(x))
}
param = list(alpha = 0.5, beta = 1.5)
background_param = list(1,1)
fit <- fit_hawkes_cbf(times = times, parameters = param,
background = background,
background_integral = background_integral,
background_parameters = background_param)
get_coefs(fit)
}
}
}
\references{
Hawkes, AG. (1971) Spectra of some self-exciting and mutually exciting point processes.
\emph{Biometrika}, \strong{58}: 83--90.
}
\seealso{
\code{\link{show_hawkes}}
}
