% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/algorithms_twosided.R
\name{rnabox}
\alias{rnabox}
\title{Recursive Neyman Algorithm for Optimal Sample Allocation Under
Box-Constraints in Stratified Sampling Schemes}
\usage{
rnabox(n, a, m = NULL, M = NULL)
}
\arguments{
\item{n}{(\code{number})\cr total sample size. A strictly positive scalar.}

\item{a}{(\code{numeric})\cr parameters \eqn{a_1, ..., a_H} of variance function
\eqn{D}. Strictly positive numbers.}

\item{m}{(\code{numeric} or \code{NULL}) \cr optional lower bounds constraints imposed
on strata sample sizes. If it is not \code{NULL}, it is required that
\code{n >= sum(m)}. Strictly positive numbers.}

\item{M}{(\code{numeric} or \code{NULL}) \cr optional upper bounds constraints imposed
on strata sample sizes. If it is not \code{NULL}, it is required that
\code{n <= sum(M)}. Strictly positive numbers.}
}
\value{
Numeric vector with optimal sample allocations in strata.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

An internal function that implements recursive Neyman optimal allocation
algorithm, \code{rnabox}. The \code{rnabox()} should not be used
directly. Instead, user function \code{\link[=dopt]{dopt()}} should be used. \cr

The \code{rnabox()} function computes:
\deqn{argmin D(x_1,...,x_H) = a^2_1/x_1 + ... + a^2_H/x_H - b,}
under the equality constraint imposed on total sample size:
\deqn{x_1 + ... + x_H = n,}
and (optionally) the following set of inequality constraints:
\deqn{m_w <= x_w <= M_w,  w = 1,...,H.}
Here, \eqn{H} denotes total number of strata, \eqn{x_1, ..., x_H} are the
strata sample sizes, and \eqn{m_w > 0, M_w > 0, w = 1, ..., H} are the lower
and upper bounds respectively, optionally imposed on sample sizes in
strata. \cr

User of \code{rnabox()} can choose which or whether the inequality constraints
will be added to the optimization problem or not. In case of no inequality
constraints to be added, \code{m} or \code{M} (or both) must be specified as \code{NULL}
(default).
If any bounds should be imposed on sample strata sizes, user must specify
these with \code{m} and \code{M} arguments.
}
\note{
For simple random sampling without replacement design in each stratum,
parameters of the variance function \eqn{D} are
\eqn{b = N_1 * S_1^2 + ... + N_H * S_H^2}, and \eqn{a_w = N_w * S_w}, where
\eqn{N_w, S_w, w = 1, ..., H}, are strata sizes and standard deviations of
a study variable in strata respectively. \cr

The \code{rnabox} is a generalization of a popular recursive
Neyman allocation procedure that is described in Remark 12.7.1 in Sarndal
et al. (1992). It is a procedure of optimal sample allocation in stratified
sampling scheme with simple random sampling without replacement design in
each stratum while not exceeding strata sizes.

Note that in case when no inequality constraints are added, the allocation
is given as a closed form expression, known as Neyman allocation:
\deqn{x_w = a_w * n / (a_1 + ... + a_H), w = 1, ..., H.}
}
\examples{
# Artificial population and the bounds that to be imposed.
N <- c(454, 10, 116, 2500, 2240, 260, 39, 3000, 2500, 400)
S <- c(0.9, 5000, 32, 0.1, 3, 5, 300, 13, 20, 7)
a <- N * S
m <- c(322, 3, 57, 207, 715, 121, 9, 1246, 1095, 294)
M <- N

# An example of a regular allocation.
n <- 6000
opt_regular <- rnabox(n, a, m, M)

# An example of a vertex allocation.
n <- 4076
opt_vertex <- rnabox(n, a, m, M)
}
\references{
To be added soon.
}
\seealso{
\code{\link[=dopt]{dopt()}}, \code{\link[=nopt]{nopt()}}, \code{\link[=sga]{sga()}}, \code{\link[=sgaplus]{sgaplus()}}, \code{\link[=coma]{coma()}}.
}
