/**
 * [IncPreDeCon.java] for Subspace MOA
 * 
 * SCStream: macro clustering algorithm
 * 
 * @author Stephan Wels
 * Data Management and Data Exploration Group, RWTH Aachen University
 */

package moa.clusterers.scstream;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

import moa.cluster.CFCluster;
import moa.cluster.Cluster;
import moa.cluster.Clustering;
import moa.clusterers.macro.AbstractMacroClusterer;
import moa.clusterers.macro.NonConvexCluster;

public class IncPreDeCon extends AbstractMacroClusterer {

	Clustering p_micro_cluster;
	Vector<PreDeConMicroCluster> mPotMicroClusters;
	private int mMinPoints;
	private double mEpsilon;
	private double mDelta;
	private int mKappa;
	private int mLambda;
	private ArrayList<PreDeConCluster> clusters;
	private boolean init;

	public IncPreDeCon(Clustering microClustering, double delta, int lambda,
			int mu, double epsilon, int kappa) {
		mPotMicroClusters = new Vector<PreDeConMicroCluster>();
		// referenz zu m_p_micro_cluster clustering
		p_micro_cluster = microClustering;
		mMinPoints = mu;
		mEpsilon = epsilon;
		mDelta = delta;
		mKappa = kappa;
		mLambda = lambda;

		for (Cluster c : microClustering.getClustering()) {
			CFCluster cf = (CFCluster) c;
			mPotMicroClusters.add(new PreDeConMicroCluster(cf, mDelta, mLambda,
					mMinPoints, mEpsilon, mKappa));
		}
	}

	public Clustering getClustering(Clustering new_pMicroClusters,
			Clustering removed_pMicroClusters) {

		if (init) {

			/**
			 * Each potential microcluster is inserted separately
			 */
			for (Cluster c : new_pMicroClusters.getClustering()) {
				CFCluster cf = (CFCluster) c;
				insert(cf);
			}

			/**
			 * Each potential microcluster is deleted separately
			 */
			for (Cluster c : removed_pMicroClusters.getClustering()) {
				CFCluster cf = (CFCluster) c;
				remove(cf);

			}

		}

		// init phase
		if (!init) {
			clusters = predecon();
			init = true;
		}

		/*
		 * Visualization stuff
		 */

		CFCluster[] res = new CFCluster[clusters.size()];
		int clusterPos = 0;
		for (PreDeConCluster cluster : clusters) {
			if (cluster.size() != 0) {
				CFCluster temp = new NonConvexCluster(cluster.get(0)
						.getCFCluster(), Convert2microclusterList(cluster));
				res[clusterPos] = temp;
				for (int i = 1; i < cluster.size(); i++) {
					res[clusterPos].add(cluster.get(i).getCFCluster());
				}
				clusterPos++;
			}
		}

		Clustering result = new Clustering(res);
		setClusterIDs(result);
		return result;

	}

	/**
	 * Insert a microcluster into an existing cluster or create a new one.
	 * 
	 * @param mc
	 */
	public void insert(CFCluster mc) {
		if (!p_micro_cluster.getClustering().contains(mc)) {
			p_micro_cluster.add(mc);
			PreDeConMicroCluster pmc = new PreDeConMicroCluster(mc, mDelta,
					mLambda, mMinPoints, mEpsilon, mKappa);
			// D = {D} u {p}
			mPotMicroClusters.add(pmc);
			// compute the subspace preference vector
			pmc.preprocessing(mPotMicroClusters);
			// pmc.epsilonWeightedNeighborhood(mPotMicroClusters); // done in
			// preprocessing

			/**
			 * update preferred dimensionality and check changes in the core
			 * member property in \N_{epsilon}(p)
			 */

			List<PreDeConMicroCluster> affected = new Vector<PreDeConMicroCluster>();
			List<PreDeConMicroCluster> neighbours = new Vector<PreDeConMicroCluster>();
			neighbours.addAll(pmc.getEpsilonNeighborhood());

			while (!neighbours.isEmpty()) {
				PreDeConMicroCluster neighbour = neighbours.get(0);
				neighbour.preprocessing(mPotMicroClusters);
				if (neighbour.mCoreChanges) {
					if (!affected.contains(neighbour)) {
						affected.add(neighbour);
						neighbour.mIsClustered = false;
						neighbour.mVisited = false;
					}
					// add only microclusters which are not processed yet
					for (PreDeConMicroCluster cl : neighbour
							.getEpsilonNeighborhood()) {
						if (!affected.contains(cl))
							neighbours.add(cl);
					}
				}
				neighbours.remove(0);
			}

			HashSet<PreDeConMicroCluster> updSeedSet = new HashSet<PreDeConMicroCluster>(
					mPotMicroClusters.size());
			for (PreDeConMicroCluster affectedCluster : affected) {
				updSeedSet.addAll(affectedCluster.getEpsilonNeighborhood());
			}

			// find a cluster where I can be added, if any exists
			HashSet<PreDeConCluster> available = new HashSet<PreDeConCluster>();
			for (PreDeConMicroCluster c : affected)
				if (c.getCluster() != null)
					available.add(c.getCluster());
			for (PreDeConMicroCluster c : pmc.getEpsilonNeighborhood())
				if (c.getCluster() != null)
					available.add(c.getCluster());
			PreDeConCluster cluster = getCluster(available);

			// compute updseeds
			LinkedList<PreDeConMicroCluster> updSeeds = new LinkedList<PreDeConMicroCluster>();
			updSeeds.addAll(updSeedSet);
			if (updSeeds.contains(pmc))
				updSeeds.remove(pmc);
			updSeeds.addFirst(pmc);

			for (PreDeConMicroCluster dmc : updSeedSet) {

				if (dmc.isCore() && !dmc.mVisited) {
					dmc.setVisited();
					List<PreDeConMicroCluster> weightedNeighbours = dmc
							.getWeightedNeighborhood();
					if (weightedNeighbours.size() > mMinPoints) {

						cluster = expandCluster(dmc, weightedNeighbours,
								cluster);

						clusters.add(cluster);
					}
				}
			}
		}
	}

	/**
	 * This function returns a new Cluster or an existing cluster. Furthermore
	 * this function merges 2 or more clusters and returns a single cluster
	 * 
	 * @param available
	 * @return
	 */
	private PreDeConCluster getCluster(HashSet<PreDeConCluster> available) {
		List<PreDeConCluster> all = new Vector<PreDeConCluster>();
		all.addAll(available);
		// No neighbour were clustered, return new cluster
		if (available.size() == 0)
			return new PreDeConCluster();
		// found a cluster where the point can be added
		if (available.size() == 1)
			return all.get(0);
		// found more clusters where the point can be added, merge them...
		if (available.size() >= 2) {
			// merging
			PreDeConCluster res = all.get(0);
			for (int i = 1; i < all.size(); i++) {
				PreDeConCluster cluster = all.get(i);
				clusters.remove(cluster);
				res.addAll(cluster);
				for (PreDeConMicroCluster pmc : cluster) {
					pmc.setCluster(res);
				}
			}
			return res;
		}
		// should not happen
		return null;
	}

	/**
	 * This function removes a microcluster from the clustering
	 * 
	 * @param cf
	 */
	public void remove(CFCluster cf) {
		PreDeConMicroCluster remove_pmc = null;
		for (PreDeConMicroCluster pmc : mPotMicroClusters) {
			if (pmc.getCFCluster().equals(cf)) {
				remove_pmc = pmc;
				break;
			}
		}

		if (remove_pmc != null) {
			// remove microcluster from final clustering
			mPotMicroClusters.remove(remove_pmc);

			// affected microclusters
			List<PreDeConMicroCluster> affected = new Vector<PreDeConMicroCluster>();
			List<PreDeConMicroCluster> visited = new Vector<PreDeConMicroCluster>();
			List<PreDeConMicroCluster> neighbours = new Vector<PreDeConMicroCluster>();
			neighbours.addAll(remove_pmc.getEpsilonNeighborhood());

			// delete remove_pmc from the neighborhood
			neighbours.remove(remove_pmc);
			// delete remove_pmc from each neighbor's neighborhood
			for (PreDeConMicroCluster n : neighbours)
				n.getEpsilonNeighborhood().remove(remove_pmc);

			// for each neighbor, update preferenced vector and check if it is
			// affected
			while (!neighbours.isEmpty()) {
				PreDeConMicroCluster neighbour = neighbours.get(0);
				neighbour.preprocessing(mPotMicroClusters);
				if (!affected.contains(neighbour))
					affected.add(neighbour);

				neighbour.mIsClustered = false;
				neighbour.mVisited = false;

				// check additionally all neighbours of affected neighbour
				for (PreDeConMicroCluster cl : neighbour
						.getEpsilonNeighborhood()) {
					if (!visited.contains(cl)) {
						neighbours.add(cl);
						visited.add(cl);
					}
				}
				neighbours.remove(0);
			}

			// remove affected microclusters from final clustering
			for (PreDeConMicroCluster c : affected)
				if (c.getCluster() != null)
					clusters.remove(c.getCluster());

			// reinsert affected microclusters
			PreDeConCluster cluster = null;
			for (PreDeConMicroCluster dmc : affected) {

				if (dmc.isCore() && !dmc.mVisited) {
					dmc.setVisited();
					List<PreDeConMicroCluster> weightedNeihbours = dmc
							.getWeightedNeighborhood();
					if (weightedNeihbours.size() > mMinPoints) {

						cluster = expandCluster(dmc, weightedNeihbours,
								new PreDeConCluster());

						clusters.add(cluster);
					}
				}
			}
		}
	}

	private ArrayList<PreDeConCluster> predecon() {

		ArrayList<PreDeConCluster> clusters = new ArrayList<PreDeConCluster>();
		for (PreDeConMicroCluster dmc : mPotMicroClusters) {
			if (dmc.isCore() && !dmc.mVisited) {
				dmc.setVisited();
				List<PreDeConMicroCluster> neighbours = dmc
						.getWeightedNeighborhood();
				if (neighbours.size() > mMinPoints) {
					PreDeConCluster cluster = expandCluster(dmc, neighbours,
							new PreDeConCluster());
					clusters.add(cluster);
				}
			} else {
				dmc.mVisited = true;
			}

		}
		return clusters;
	}

	private PreDeConCluster expandCluster(PreDeConMicroCluster pmc,
			List<PreDeConMicroCluster> neighbours, PreDeConCluster cluster) {

		if (!pmc.isClustered()) {
			cluster.add(pmc);
			pmc.setClustered();
			pmc.setCluster(cluster);
		}

		while (!neighbours.isEmpty()) {
			PreDeConMicroCluster mc = neighbours.get(0);
			neighbours.remove(0);
			if (!mc.isVisited() && mc.isCore()) {
				mc.setVisited();
				List<PreDeConMicroCluster> neighbours2 = mc
						.getWeightedNeighborhood();
				if (neighbours2.size() >= mMinPoints) {
					while (!neighbours2.isEmpty()) {
						PreDeConMicroCluster temp = neighbours2.get(0);
						neighbours2.remove(0);
						if (!temp.isVisited()) {
							neighbours.add(temp);
						}
					}
					neighbours.addAll(neighbours2);
					if (!mc.isClustered()) {
						mc.setClustered();
						mc.setCluster(cluster);
						cluster.add(mc);
					}
				}
			}
		}
		return cluster;
	}

	/**
	 * for visualization
	 * 
	 * @param cluster
	 * @return
	 */
	private List<CFCluster> Convert2microclusterList(PreDeConCluster cluster) {
		List<CFCluster> cfCluster = new Vector<CFCluster>();
		for (PreDeConMicroCluster d : cluster) {
			cfCluster.add(d.getCFCluster());
		}
		return cfCluster;
	}

	@Override
	public Clustering getClustering(Clustering microClusters) {
		// TODO Auto-generated method stub
		return null;
	}

}
