/**
 * [IncPreDeConOpt.java] for Subspace MOA
 * 
 * SCStream: macro clustering algorithm for the optimized version
 * 
 * @author Stephan Wels
 * Data Management and Data Exploration Group, RWTH Aachen University
 */

package moa.clusterers.scstream;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

import moa.cluster.CFCluster;
import moa.cluster.Cluster;
import moa.cluster.Clustering;
import moa.clusterers.macro.AbstractMacroClusterer;
import moa.clusterers.macro.NonConvexCluster;

public class IncPreDeConOpt extends AbstractMacroClusterer {

	Clustering p_micro_cluster;
	Vector<PreDeConMicroCluster> mPotMicroClusters;
	private int mMinPoints;
	private double mEpsilon;
	private double mDelta;
	private int mKappa;
	private int mLambda;
	private ArrayList<PreDeConCluster> clusters;

	public IncPreDeConOpt(Clustering microClustering, double delta, int lambda,
			int mu, double epsilon, int kappa) {
		mPotMicroClusters = new Vector<PreDeConMicroCluster>();
		// referenz zu m_p_micro_cluster clustering
		p_micro_cluster = microClustering;
		mMinPoints = mu;
		mEpsilon = epsilon;
		mDelta = delta;
		mKappa = kappa;
		mLambda = lambda;

		for (Cluster c : microClustering.getClustering()) {
			CFCluster cf = (CFCluster) c;
			mPotMicroClusters.add(new PreDeConMicroCluster(cf, mDelta, mLambda,
					mMinPoints, mEpsilon, mKappa));
		}
	}

	boolean init;

	public Clustering getClustering(Clustering new_pMicroClusters,
			Clustering removed_pMicroClusters) {

		// RECLUSTERING
		if (init) {

			int maxSizeInsert = new_pMicroClusters.getClustering().size();
			int maxSizeDelete = removed_pMicroClusters.getClustering().size();

			// Insertion: search all affected microclusters by the new inserted
			// microclusters.
			HashSet<PreDeConMicroCluster> affectedSetbyInsert = new HashSet<PreDeConMicroCluster>(
					maxSizeInsert);

			for (Cluster c : new_pMicroClusters.getClustering()) {
				CFCluster cf = (CFCluster) c;
				affectedSetbyInsert.addAll(getAffectedByInsert(cf));
			}
			insertAffectedInsertSet(affectedSetbyInsert);

			// Deletion: search all affected microclusters by the deleted
			// microclusters.
			HashSet<PreDeConMicroCluster> affectedSetByDelete = new HashSet<PreDeConMicroCluster>(
					maxSizeDelete);
			LinkedList<PreDeConMicroCluster> mcToRemoveFromClustering = new LinkedList<PreDeConMicroCluster>();

			for (Cluster c : removed_pMicroClusters.getClustering()) {
				CFCluster cf = (CFCluster) c;

				PreDeConMicroCluster remove_pmc = null;
				for (PreDeConMicroCluster pmc : mPotMicroClusters) {
					if (pmc.getCFCluster().equals(cf)) {
						remove_pmc = pmc;
						break;
					}
				}
				if (remove_pmc != null) {
					mcToRemoveFromClustering.add(remove_pmc);
					affectedSetByDelete
							.addAll(getAffectedByRemoving(remove_pmc));
				}
				else{
					System.out.println("should not happen");
					System.out.println(c.toString());
				}
			}

			// Right now, the removeSet contains all the Affected microcluster
			// from the microclusters which are "Deleted" AND the microclusters
			// which are "Deleted" themselves.
			affectedSetByDelete.removeAll(mcToRemoveFromClustering);
			reinsertAffectedDeletionSet(affectedSetByDelete);

			// RECLUSTERING DONE

		}

		// initialization phase
		if (!init) {
			clusters = predecon();
			init = true;
		}

		/*
		 * Visualization stuff
		 */
		CFCluster[] res = new CFCluster[clusters.size()];
		int clusterPos = 0;
		for (PreDeConCluster cluster : clusters) {
			if (cluster.size() != 0) {
				CFCluster temp = new NonConvexCluster(cluster.get(0)
						.getCFCluster(), Convert2microclusterList(cluster));
				res[clusterPos] = temp;
				for (int i = 1; i < cluster.size(); i++) {
					res[clusterPos].add(cluster.get(i).getCFCluster());
				}
				clusterPos++;
			}
		}

		Clustering result = new Clustering(res);
		setClusterIDs(result);
		return result;
	}

	private void insertAffectedInsertSet(HashSet<PreDeConMicroCluster> insertSet) {
		LinkedList<PreDeConMicroCluster> updSeeds = new LinkedList<PreDeConMicroCluster>();
		updSeeds.addAll(insertSet);

		PreDeConCluster cluster;

		for (PreDeConMicroCluster dmc : updSeeds) {

			if (dmc.isCore() && !dmc.mVisited) {
				dmc.setVisited();
				List<PreDeConMicroCluster> weightedNeihbours = dmc
						.getWeightedNeighborhood();
				if (weightedNeihbours.size() > mMinPoints) {

					cluster = expandCluster_ModifiedForMergeCase(dmc,
							weightedNeihbours, new PreDeConCluster());

					clusters.add(cluster);
				}
			}
		}
	}

	public List<PreDeConMicroCluster> getAffectedByInsert(CFCluster mc) {
		if (!p_micro_cluster.getClustering().contains(mc)) {
			p_micro_cluster.add(mc);
			PreDeConMicroCluster pmc = new PreDeConMicroCluster(mc, mDelta,
					mLambda, mMinPoints, mEpsilon, mKappa);
			// D = {D} u {p}
			mPotMicroClusters.add(pmc);
			// compute the subspace preference vector
			pmc.preprocessing(mPotMicroClusters);

			/**
			 * update preferred dimensionality and check changes in the core
			 * member property in \N_{epsilon}(p)
			 */

			List<PreDeConMicroCluster> affected = new Vector<PreDeConMicroCluster>();
			List<PreDeConMicroCluster> neighbours = new Vector<PreDeConMicroCluster>();
			neighbours.addAll(pmc.getEpsilonNeighborhood());

			while (!neighbours.isEmpty()) {
				PreDeConMicroCluster neighbour = neighbours.get(0);
				neighbour.preprocessing(mPotMicroClusters);
				if (neighbour.mCoreChanges) {
					if (!affected.contains(neighbour)) {
						affected.add(neighbour);
						neighbour.mIsClustered = false;
						neighbour.mVisited = false;
					}
					// add only microclusters which are not processed yet
					for (PreDeConMicroCluster cl : neighbour
							.getEpsilonNeighborhood()) {
						if (!affected.contains(cl))
							neighbours.add(cl);
					}
				}
				neighbours.remove(0);
			}

			HashSet<PreDeConMicroCluster> updSeedSet = new HashSet<PreDeConMicroCluster>(
					mPotMicroClusters.size());
			for (PreDeConMicroCluster affectedCluster : affected) {
				updSeedSet.addAll(affectedCluster.getEpsilonNeighborhood());
			}

			// compute updseeds
			LinkedList<PreDeConMicroCluster> updSeeds = new LinkedList<PreDeConMicroCluster>();
			updSeeds.addAll(updSeedSet);
			if (updSeeds.contains(pmc))
				updSeeds.remove(pmc);
			updSeeds.addFirst(pmc);

			return updSeeds;
		}
		return null;
	}

	public void reinsertAffectedDeletionSet(
			HashSet<PreDeConMicroCluster> removeSet) {
		LinkedList<PreDeConMicroCluster> updSeeds = new LinkedList<PreDeConMicroCluster>();
		updSeeds.addAll(removeSet);

		for (PreDeConMicroCluster mc : updSeeds) {
			mc.mIsClustered = false;
			mc.mVisited = false;
			if (mc.getCluster() != null)
				clusters.remove(mc.getCluster());
			mc.setCluster(null);
		}

		// reinsert affected microclusters
		PreDeConCluster cluster = null;
		for (PreDeConMicroCluster dmc : updSeeds) {

			if (dmc.isCore() && !dmc.mVisited) {
				dmc.setVisited();
				List<PreDeConMicroCluster> weightedNeihbours = dmc
						.getWeightedNeighborhood();
				if (weightedNeihbours.size() > mMinPoints) {

					cluster = expandCluster(dmc, weightedNeihbours,
							new PreDeConCluster());

					clusters.add(cluster);
				}
			}
		}

	}

	public List<PreDeConMicroCluster> getAffectedByRemoving(
			PreDeConMicroCluster remove_pmc) {
		// remove microcluster from final clustering
		mPotMicroClusters.remove(remove_pmc);

		// affected microclusters
		List<PreDeConMicroCluster> affected = new Vector<PreDeConMicroCluster>();
		List<PreDeConMicroCluster> visited = new Vector<PreDeConMicroCluster>();
		List<PreDeConMicroCluster> neighbours = new Vector<PreDeConMicroCluster>();
		neighbours.addAll(remove_pmc.getEpsilonNeighborhood());

		// delete remove_pmc from the neighborhood
		neighbours.remove(remove_pmc);
		// delete remove_pmc from each neighbor's neighborhood
		for (PreDeConMicroCluster n : mPotMicroClusters)
			// neighbours)
			n.getEpsilonNeighborhood().remove(remove_pmc);

		// for each neighbor, update preferenced vector and check if it is
		// affected
		while (!neighbours.isEmpty()) {
			PreDeConMicroCluster neighbour = neighbours.get(0);
			neighbour.preprocessing(mPotMicroClusters);
			if (!affected.contains(neighbour))
				affected.add(neighbour);

			neighbour.mIsClustered = false;
			neighbour.mVisited = false;

			// check additionally all neighbours of affected neighbour
			for (PreDeConMicroCluster cl : neighbour.getEpsilonNeighborhood()) {
				if (!visited.contains(cl)) {
					neighbours.add(cl);
					visited.add(cl);
				}
			}
			neighbours.remove(0);
		}

		// remove affected microclusters from final clustering
		for (PreDeConMicroCluster c : affected)
			if (c.getCluster() != null) {
				clusters.remove(c.getCluster());
				c.setCluster(null);
			}
		return affected;
	}

	private ArrayList<PreDeConCluster> predecon() {

		ArrayList<PreDeConCluster> clusters = new ArrayList<PreDeConCluster>();
		for (PreDeConMicroCluster dmc : mPotMicroClusters) {
			if (dmc.isCore() && !dmc.mVisited) {
				dmc.setVisited();
				List<PreDeConMicroCluster> neighbours = dmc
						.getWeightedNeighborhood();
				if (neighbours.size() > mMinPoints) {
					PreDeConCluster cluster = expandCluster(dmc, neighbours,
							new PreDeConCluster());
					clusters.add(cluster);
				}
			} else {
				dmc.mVisited = true;
			}

		}
		return clusters;
	}

	private PreDeConCluster expandCluster(PreDeConMicroCluster pmc,
			List<PreDeConMicroCluster> neighbours, PreDeConCluster cluster) {

		if (!pmc.isClustered()) {
			cluster.add(pmc);
			pmc.setClustered();
			pmc.setCluster(cluster);
		}
		int i = 0;
		while (!neighbours.isEmpty()) {
			PreDeConMicroCluster mc = neighbours.get(0);
			neighbours.remove(0);
			if (!mc.isVisited() && mc.isCore()) {
				mc.setVisited();
				List<PreDeConMicroCluster> neighbours2 = mc
						.getWeightedNeighborhood();
				if (neighbours2.size() >= mMinPoints) {
					while (!neighbours2.isEmpty()) {
						PreDeConMicroCluster temp = neighbours2.get(0);
						neighbours2.remove(0);
						if (!temp.isVisited()) {
							neighbours.add(temp);
						}
					}
					neighbours.addAll(neighbours2);
					if (!mc.isClustered()) {
						mc.setClustered();
						mc.setCluster(cluster);
						cluster.add(mc);
					}

				}
			}
		}
		return cluster;
	}

	/**
	 * modified expandCluster function for the insert case. It might happen that
	 * 2 clusters (or more) have to be merged. check if any density connected
	 * neighbour belong to another cluster. Since both are directly density
	 * reachable, they have to belong to the same cluster, i.e. clusters have to
	 * be merged
	 * 
	 * @param pmc
	 * @param neighbours
	 * @param cluster
	 * @return
	 */
	private PreDeConCluster expandCluster_ModifiedForMergeCase(
			PreDeConMicroCluster pmc, List<PreDeConMicroCluster> neighbours,
			PreDeConCluster cluster) {

		// list of all clusters which have to be merged;
		Vector<PreDeConCluster> merge = new Vector<PreDeConCluster>();
		merge.add(cluster);

		if (!pmc.isClustered()) {
			cluster.add(pmc);
			pmc.setClustered();
			pmc.setCluster(cluster);
		}

		int i = 0;
		while (!neighbours.isEmpty()) {
			PreDeConMicroCluster mc = neighbours.get(0);
			neighbours.remove(0);

			if (mc.getCluster() != null && !merge.contains(mc.getCluster())) {
				merge.add(mc.getCluster());
			}

			if (!mc.isVisited() && mc.isCore()) {
				mc.setVisited();

				List<PreDeConMicroCluster> neighbours2 = mc
						.getWeightedNeighborhood();
				if (neighbours2.size() >= mMinPoints) {
					while (!neighbours2.isEmpty()) {
						PreDeConMicroCluster temp = neighbours2.get(0);
						neighbours2.remove(0);
						if (!temp.isVisited()) {
							neighbours.add(temp);
						}
					}
					neighbours.addAll(neighbours2);
					if (!mc.isClustered()) {
						mc.setClustered();
						mc.setCluster(cluster);
						cluster.add(mc);
					}
				}
			}
		}

		if (merge.size() == 1)
			return cluster;
		if (merge.size() >= 2) {
			// merging
			PreDeConCluster res = merge.get(0);
			for (int c = 1; c < merge.size(); c++) {
				PreDeConCluster toMerge = merge.get(c);
				clusters.remove(toMerge);
				res.addAll(toMerge);
				for (PreDeConMicroCluster microcluster : toMerge) {
					microcluster.setCluster(res);
				}
			}
			return res;
		}
		// Should not happen
		return null;
	}

	/**
	 * for visualization
	 * 
	 * @param cluster
	 * @return
	 */
	private List<CFCluster> Convert2microclusterList(PreDeConCluster cluster) {
		List<CFCluster> cfCluster = new Vector<CFCluster>();
		for (PreDeConMicroCluster d : cluster) {
			cfCluster.add(d.getCFCluster());
		}
		return cfCluster;
	}

	@Override
	public Clustering getClustering(Clustering microClusters) {
		// TODO Auto-generated method stub
		return null;
	}

}
