% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evdsi.R
\name{evdsi}
\alias{evdsi}
\title{Expected value of the decision given survey information}
\usage{
evdsi(
  site_data,
  feature_data,
  site_detection_columns,
  site_n_surveys_columns,
  site_probability_columns,
  site_management_cost_column,
  site_survey_scheme_column,
  site_survey_cost_column,
  feature_survey_column,
  feature_survey_sensitivity_column,
  feature_survey_specificity_column,
  feature_model_sensitivity_column,
  feature_model_specificity_column,
  feature_target_column,
  total_budget,
  site_management_locked_in_column = NULL,
  site_management_locked_out_column = NULL,
  prior_matrix = NULL
)
}
\arguments{
\item{site_data}{\code{\link[sf:sf]{sf::sf()}} object with site data.}

\item{feature_data}{\code{\link[base:data.frame]{base::data.frame()}} object with feature data.}

\item{site_detection_columns}{\code{character} names of \code{numeric}
columns in the argument to \code{site_data} that contain the proportion of
surveys conducted within each site that detected each feature.
Each column should correspond to a different feature, and contain
a proportion value (between zero and one). If a site has
not previously been surveyed, a value of zero should be used.}

\item{site_n_surveys_columns}{\code{character} names of \code{numeric}
columns in the argument to \code{site_data} that contain the total
number of surveys conducted for each each feature within each site.
Each column should correspond to a different feature, and contain
a non-negative integer number (e.g. 0, 1, 2, 3). If a site has
not previously been surveyed, a value of zero should be used.}

\item{site_probability_columns}{\code{character} names of \code{numeric}
columns in the argument to \code{site_data} that contain modelled
probabilities of occupancy for each feature in each site.
Each column should correspond to a different feature, and contain
probability data (values between zero and one). No missing (\code{NA})
values are permitted in these columns.}

\item{site_management_cost_column}{\code{character} name of column in the
argument to \code{site_data} that contains costs for managing each
site for conservation. This column should have \code{numeric} values that
are equal to or greater than zero. No missing (\code{NA}) values are
permitted in this column.}

\item{site_survey_scheme_column}{\code{character} name of \code{logical}
(\code{TRUE} / \code{FALSE}) column in the argument to \code{site_data}
that indicates which sites are selected in the scheme or not.
No missing \code{NA} values are permitted. Additionally, only sites
that are missing data can be selected or surveying (as per the
argument to \code{site_detection_columns}).}

\item{site_survey_cost_column}{\code{character} name of column in the
argument to  \code{site_data} that contains costs for surveying each
site. This column should have \code{numeric} values that are equal to
or greater than zero. No missing (\code{NA}) values are permitted in this
column.}

\item{feature_survey_column}{\code{character} name of the column in the
argument to \code{feature_data} that contains \code{logical} (\code{TRUE} /
\code{FALSE}) values indicating if the feature will be surveyed in
the planned surveys or not. Note that considering additional features will
rapidly increase computational burden, and so it is only recommended to
consider features that are of specific conservation interest.
No missing (\code{NA}) values are permitted in this column.}

\item{feature_survey_sensitivity_column}{\code{character} name of the
column in the argument to \code{feature_data} that contains
probability of future surveys correctly detecting a presence of each
feature in a given site (i.e. the sensitivity of the survey methodology).
This column should have \code{numeric} values that are between zero and
one. No missing (\code{NA}) values are permitted in this column.}

\item{feature_survey_specificity_column}{\code{character} name of the
column in the argument to \code{feature_data} that contains
probability of future surveys correctly detecting an absence of each
feature in a given site (i.e. the specificity of the survey methodology).
This column should have \code{numeric} values that are between zero and
one. No missing (\code{NA}) values are permitted in this column.}

\item{feature_model_sensitivity_column}{\code{character} name of the
column in the argument to \code{feature_data} that contains
probability of the initial models correctly predicting a presence of each
feature in a given site (i.e. the sensitivity of the models).
This column should have \code{numeric} values that are between zero and
one. No missing (\code{NA}) values are permitted in this column.
This should ideally be calculated using
\code{\link[=fit_xgb_occupancy_models]{fit_xgb_occupancy_models()}} or
\code{\link[=fit_hglm_occupancy_models]{fit_hglm_occupancy_models()}}.}

\item{feature_model_specificity_column}{\code{character} name of the
column in the argument to \code{feature_data} that contains
probability of the initial models correctly predicting an absence of each
feature in a given site (i.e. the specificity of the models).
This column should have \code{numeric} values that are between zero and
one. No missing (\code{NA}) values are permitted in this column.
This should ideally be calculated using
\code{\link[=fit_xgb_occupancy_models]{fit_xgb_occupancy_models()}} or
\code{\link[=fit_hglm_occupancy_models]{fit_hglm_occupancy_models()}}.}

\item{feature_target_column}{\code{character} name of the column in the
argument to \code{feature_data} that contains the \eqn{target}
values used to parametrize the conservation benefit of managing of each
feature.
This column should have \code{numeric} values that
are equal to or greater than zero. No missing (\code{NA}) values are
permitted in this column.}

\item{total_budget}{\code{numeric} maximum expenditure permitted
for conducting surveys and managing sites for conservation.}

\item{site_management_locked_in_column}{\code{character} name of the column
in the argument to \code{site_data} that contains \code{logical}
(\code{TRUE} / \code{FALSE}) values indicating which sites should
be locked in for (\code{TRUE}) being managed for conservation or
(\code{FALSE}) not. No missing (\code{NA}) values are permitted in this
column. This is useful if some sites have already been earmarked for
conservation, or if some sites are already being managed for conservation.
Defaults to \code{NULL} such that no sites are locked in.}

\item{site_management_locked_out_column}{\code{character} name of the column
in the argument to \code{site_data} that contains \code{logical}
(\code{TRUE} / \code{FALSE}) values indicating which sites should
be locked out for (\code{TRUE}) being managed for conservation or
(\code{FALSE}) not. No missing (\code{NA}) values are permitted in this
column. This is useful if some sites could potentially be surveyed
to improve model predictions even if they cannot be managed for
conservation. Defaults to \code{NULL} such that no sites are locked out.}

\item{prior_matrix}{\code{numeric} \code{matrix} containing
the prior probability of each feature occupying each site.
Rows correspond to features, and columns correspond to sites.
Defaults to \code{NULL} such that prior data is calculated automatically
using \code{\link[=prior_probability_matrix]{prior_probability_matrix()}}.}
}
\value{
\code{numeric} value.
}
\description{
Calculate the \emph{expected value of the management decision
given survey information}. This metric describes the value of the management
decision that is expected when the decision maker surveys a
set of sites to help inform the decision.
}
\details{
This function calculates the expected value and does not
use approximation methods. As such, this function can only be applied
to very small problems.
}
\examples{
# set seeds for reproducibility
library(RandomFields)
set.seed(123)
RFoptions(seed = 123)

# simulate data
site_data <- simulate_site_data(n_sites = 30, n_features = 2, prop = 0.1)
feature_data <- simulate_feature_data(n_features = 2, prop = 1)
feature_data$target <- c(10, 10)

# preview simulated data
print(site_data)
print(feature_data)

# set total budget for managing sites for conservation
 # (i.e. 50\% of the cost of managing all sites)
total_budget <- sum(site_data$management_cost) * 0.5

# create a survey scheme that samples the first two sites that
# are missing data
site_data$survey_site <- FALSE
site_data$survey_site[which(site_data$n1 < 0.5)[1:2]] <- TRUE

# calculate expected value of management decision given the survey
# information using exact method
ev_survey <- evdsi(
  site_data, feature_data,
  c("f1", "f2"), c("n1", "n2"), c("p1", "p2"),
  "management_cost", "survey_site",
  "survey_cost", "survey", "survey_sensitivity", "survey_specificity",
  "model_sensitivity", "model_specificity",
  "target", total_budget)

# print value
print(ev_survey)
}
\seealso{
\code{\link[=prior_probability_matrix]{prior_probability_matrix()}}.
}
